# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Fix a seed for reproducibility
np.random.seed(42)

# Generate bimodal distribution for Heart Rate data
# Simulate heart rates for two groups (Group A and Group B)
heart_rate_group_a = np.random.normal(loc=70, scale=10, size=400)  # Normal heart rates
heart_rate_group_b = np.random.normal(loc=110, scale=15, size=400) # Elevated heart rates
heart_rate_data = np.concatenate([heart_rate_group_a, heart_rate_group_b])

# X-axis values for KDE
xs = np.linspace(40, 150, 200)

# Axes Limits and Labels
title = "KDE Plot of Heart Rate Distribution for Two Groups"
xlabel_value = "Heart Rate (bpm)"
ylabel_value = "Density"
legend_label = "Heart Rate Density for Groups A and B"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

xticks_values = [40, 70, 100, 130, 150]
xticklabels = ["40", "70", "100", "130", "150"]
yticks_values = [0, 0.007, 0.014, 0.021, 0.028]
yticklabels = ["0", "0.007", "0.014", "0.021", "0.028"]
xlim_values = [40, 150]
ylim_values = [0, 0.03]

legend_location = 'upper right'
legend_fontsize = 12
title_fontsize = 14
title_weight = 'bold'
label_fontsize = 12

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(heart_rate_data)
density.covariance_factor = lambda: 0.3
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#ff9999", edgecolor="dodgerblue")

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=12)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=12)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_location, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_14.pdf", bbox_inches="tight")