# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(24)

# Generate sample temperature data
spring_temp = np.random.normal(loc=15, scale=5, size=1000)  # in degrees Celsius
summer_temp = np.random.normal(loc=25, scale=6, size=1000)  # in degrees Celsius
winter_temp = np.random.normal(loc=5, scale=7, size=1000)  # in degrees Celsius

# Labels and texts
labels = ["Spring Temperature", "Summer Temperature", "Winter Temperature"]
avxlabel = "Average Spring Temperature"
xlabel = "Temperature (°C)"
ylabel = "Density"
title = "Seasonal Temperature Data Density Distributions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

grid_linestyle = "-"
grid_linewidth = 0.5
grid_alpha = 0.7
line_style = "--"
line_width = 2
fill_alpha = 0.25

# Plot the density plots
for data, color, label in zip(
    [spring_temp, summer_temp, winter_temp],
    ["blue", "orange", "green"],
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 5, max(data) + 5, 200)
    density.covariance_factor = lambda: 0.25  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_style,  # dashed line
        linewidth=line_width  # slightly thinner lines
    )
    plt.fill_between(xs, density(xs), color=color, alpha=fill_alpha)  # stronger fill for shade

# Plot a vertical line indicating the average spring temperature
plt.axvline(x=np.mean(spring_temp), color="red", linestyle=line_style, linewidth=line_width, label=avxlabel)

# Set labels and title
ax.set_xlim(min(min(spring_temp), min(summer_temp), min(winter_temp)) - 5,
            max(max(spring_temp), max(summer_temp), max(winter_temp)) + 5)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid with a thicker linestyle
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_16.pdf", bbox_inches="tight")