# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(123)

# Generate new sample data for weekly sales
sales_high = np.random.normal(loc=1500, scale=300, size=1000)  # High season
sales_med = np.random.normal(loc=1000, scale=250, size=1000)  # Medium season
sales_low = np.random.normal(loc=700, scale=200, size=1000)  # Low season

# Labels for the new data
labels = ["High Season", "Medium Season", "Low Season"]
avxlabel = "Average Weekly Sales"
xlabel = "Weekly Sales ($)"
ylabel = "Density"
title = "Weekly Sales Distribution by Season"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure and axis
fig, ax = plt.subplots(figsize=(9, 6))

grid_linestyle = "--"
grid_alpha = 0.6
average_sales = 1066.67
xlim_min = 200
xlim_max = 2300
xticks_step = 200

# Define colors for the different sales seasons
colors = ["#d62728", "#9467bd", "#8c564b"]

# Plot the density plots
for data, color, label in zip([sales_high, sales_med, sales_low], colors, labels):
    density = gaussian_kde(data)
    xs = np.linspace(xlim_min, xlim_max, 200)
    density.covariance_factor = lambda: 0.4
    density._compute_covariance()
    plt.fill_between(xs, density(xs), color=color, alpha=0.3, label=label)

# Plot the average weekly sales line
plt.axvline(x=average_sales, color="green", linestyle="--", label=avxlabel)

# Set chart labels and title
ax.set_xlim(xlim_min, xlim_max)
ax.set_xticks(np.arange(xlim_min, xlim_max + 1, xticks_step))
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
plt.title(title)

# Show grid
plt.grid(True, linestyle=grid_linestyle, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_17.pdf", bbox_inches="tight")