# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Reproducibility
np.random.seed(3)
# Simulated smartphone battery life data (in hours)
battery_life1 = np.random.normal(loc=15, scale=2, size=1000)
battery_life2 = np.random.normal(loc=12, scale=3, size=1000)
battery_life3 = np.random.normal(loc=20, scale=1.5, size=1000)

# Compute density for each dataset
density1 = gaussian_kde(battery_life1)
density2 = gaussian_kde(battery_life2)
density3 = gaussian_kde(battery_life3)
xs = np.linspace(5, 25, 400)
ys1 = density1(xs)
ys2 = density2(xs)
ys3 = density3(xs)

# Labels
labels = ["Model A", "Model B", "Model C"]
xlabel = "Battery Life (hours)"
ylabel = "Density"
title = "Density Plot of Smartphone Battery Life"
peak_annotation_template = "Peak: {:.1f}"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

grid_linestyle = "--"
line_linestyle = "-."
line_width = 2
arrowprops = dict(facecolor='black', shrink=0.05)

# Fill between x for density regions with optimization
plt.fill_between(xs, ys1, color="lightcoral", alpha=0.5, label=labels[0])
plt.fill_between(xs, ys2, color="lightseagreen", alpha=0.5, label=labels[1])
plt.fill_between(xs, ys3, color="lightsalmon", alpha=0.5, label=labels[2])

# Plot lines for densities
plt.plot(xs, ys1, color="maroon", linestyle=line_linestyle, linewidth=line_width)
plt.plot(xs, ys2, color="darkcyan", linestyle=line_linestyle, linewidth=line_width)
plt.plot(xs, ys3, color="peru", linestyle=line_linestyle, linewidth=line_width)

# Set labels and title (if any)
ax.set_xlim(5, 27.5)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Annotations for peaks
peak1 = xs[np.argmax(ys1)]
peak2 = xs[np.argmax(ys2)]
peak3 = xs[np.argmax(ys3)]
plt.annotate(peak_annotation_template.format(peak1), xy=(peak1, max(ys1)), xytext=(peak1+1, max(ys1)-0.02),
             arrowprops=arrowprops)
plt.annotate(peak_annotation_template.format(peak2), xy=(peak2, max(ys2)), xytext=(peak2+1, max(ys2)+0.02),
             arrowprops=arrowprops)
plt.annotate(peak_annotation_template.format(peak3), xy=(peak3, max(ys3)), xytext=(peak3+1, max(ys3)-0.02),
             arrowprops=arrowprops)

# Show grid
plt.grid(True, linestyle=grid_linestyle)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_20.pdf", bbox_inches="tight")