# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
# Create a directed graph
communication_network = nx.DiGraph()

# Add nodes with their respective office types and colors
office_nodes = {
    0: ("New York Headquarters", "navy"),
    1: ("Chicago Regional Office", "lightblue"),
    2: ("Los Angeles Regional Office", "lightgreen"),
    3: ("Chicago Branch Office 1", "orange"),
    4: ("Chicago Branch Office 2", "pink"),
    5: ("LA Branch Office 1", "purple"),
    6: ("LA Branch Office 2", "red"),
}
for node, (office, color) in office_nodes.items():
    communication_network.add_node(node, office=office, color=color)

# Add edges with bandwidth capacities
network_links = [(0, 1, "500Mbps"), (0, 2, "600Mbps"), (1, 3, "200Mbps"), 
                 (1, 4, "250Mbps"), (2, 5, "300Mbps"), (2, 6, "350Mbps")]
for u, v, capacity in network_links:
    communication_network.add_edge(u, v, capacity=capacity)

plot_title = "Corporate Office Communication Network Map"
legend_title = "Office Types"

# Define node positions in a circular layout
node_positions = nx.circular_layout(communication_network)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute and label them
office_colors = [communication_network.nodes[node]["color"] for node in communication_network.nodes]
nx.draw_networkx_nodes(communication_network, node_positions, node_color=office_colors, node_size=700)
nx.draw_networkx_labels(communication_network, node_positions, labels={node: communication_network.nodes[node]["office"] for node in communication_network.nodes}, font_size=10)

# Draw edges with capacity labels
nx.draw_networkx_edges(communication_network, node_positions, arrowstyle='->', arrowsize=20)
edge_labels = {(u, v): communication_network[u][v]["capacity"] for u, v in communication_network.edges}
nx.draw_networkx_edge_labels(communication_network, node_positions, edge_labels=edge_labels, font_size=8)

# Add a title
plt.title(plot_title, fontsize=18)

# Add legend manually
legend_labels = {color: office_type for _, (office_type, color) in office_nodes.items()}
for color in set(office_colors):
    ax.plot([], [], color=color, label=legend_labels[color], marker='o', markersize=10, linestyle='')

# Removing the axis
plt.axis("off")

# Adding legend to the plot
plt.legend(title=legend_title, loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_17.pdf", bbox_inches="tight")