# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
import numpy as np
np.random.seed(0)  # Ensuring reproducibility
# Create a random graph to represent a computer network
computer_network_graph = nx.random_geometric_graph(20, 0.3)

# Position nodes using the Kamada-Kawai layout for better visualization
node_positions = nx.kamada_kawai_layout(computer_network_graph)

# Node type assignment: 50% servers (blue), rest workstations (green)
node_colors = []
for node in computer_network_graph.nodes():
    if np.random.rand() > 0.5:
        node_colors.append('blue')  # Server
    else:
        node_colors.append('orange')  # Workstation

# Edges: Mark 30% of the edges as monitored (dashed lines, highlighted)
all_edges = list(computer_network_graph.edges())
num_edges_monitored = int(len(all_edges) * 0.3)
monitored_edges_indices = np.random.choice(len(all_edges), size=num_edges_monitored, replace=False)
edges_monitored = [all_edges[i] for i in monitored_edges_indices]

updated_title = "Computer Network with Devices and Connection Status"
labels = ["Workstation", "Server", "Active Connection", "Monitored Connection"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw the nodes with respective colors
nx.draw_networkx_nodes(computer_network_graph, node_positions, node_size=100, node_color=node_colors, ax=ax)

# Draw all edges
nx.draw_networkx_edges(computer_network_graph, node_positions, ax=ax, edge_color="gray", alpha=0.5)

# Draw monitored edges with a different style (dashed)
nx.draw_networkx_edges(computer_network_graph, node_positions, ax=ax, edgelist=edges_monitored, style='dashed', edge_color="orange")

# Add a legend
legend_elements = [
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor='orange', markersize=10, label=labels[0]),
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor='blue', markersize=10, label=labels[1]),
    plt.Line2D([0], [0], color='gray', lw=2, alpha=0.5, label=labels[2]),
    plt.Line2D([0], [0], color='red', linestyle='--', linewidth=2, label=labels[3])
]

ax.legend(handles=legend_elements, loc='upper right')

# Remove axis
plt.axis("off")

# Add a title
plt.title(updated_title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_9.pdf", bbox_inches="tight")