import numpy as np; np.random.seed(0); np.random.seed(0)
import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
# Defining the colormap from yellow to blue
cmap = plt.cm.coolwarm_r

# Sample data: a 5x6 grid, reflecting business performance metrics
data = np.array(
    [
        [250.3, 120.0, 90.9, 70.0, 134.4, 163.3],
        [492.2, 110.2, 60.6, 78.8, 173.3, 139.9],
        [508.8, 123.3, 90.9, 62.2, 155.5, 143.3],
        [760.0, 21.1, 50.5, 41.1, 81.1, 92.2],
        [157.7, 281.1, 26.6, 145.5, 286.6, 105.5],
    ]
)

# X and Y labels for business departments
x_labels = ["Sales", "Marketing", "IT", "HR", "Finance", "Admin"]
y_labels = ["Q1", "Q2", "Q3", "Q4", "Q5"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=20, vmax=1000)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle with a fixed size
        circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
        ax.add_artist(circle)

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f}", ha="center", va="center", color="black")

# Set labels for x and y axes
ax.set_xticks(range(len(x_labels)))
ax.set_xticklabels(x_labels, ha="center")
ax.set_yticks(range(len(y_labels)))
ax.set_yticklabels(y_labels, va="center")

# Adding titles for the axes
ax.set_xlabel("Departments", fontsize=14)
ax.set_ylabel("Quarters", fontsize=14)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[20, 100, 500, 1000], orientation="vertical")
cbar.ax.set_yticklabels(["20", "100", "500", "1000"])

# ===================
# Part 4: Saving Output
# ===================
# Save the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig('heatmap_10.pdf', bbox_inches='tight')
