# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
# Updated for a new domain: Furniture and Appliance Brands Performance and Usability Bias
furniture_brands = ["IKEA", "Qumeijia", "Shangpinzhaipei"]
market_scores_furniture = [11.0, 9.5, 7.0]
market_bias_furniture = [+1.2, +2.8, +3.0]
usability_scores_furniture = [6.5, 7.5, 5.0]
usability_bias_furniture = [-1.5, -1.8, -2.0]
furniture_labels = ["Furniture Market", "Usability Bias"]

appliance_brands = ["Haier", "Midea", "Siemens"]
market_scores_appliance = [10.2, 9.8, 8.5]
market_bias_appliance = [+0.5, +2.0, +1.5]
usability_scores_appliance = [8.0, 7.8, 7.5]
usability_bias_appliance = [-1.0, -0.8, -1.5]
appliance_labels = ["Washing Machines Market", "Usability Bias"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (furniture brands)
for i, brand in enumerate(furniture_brands):
    # Market bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(market_scores_furniture[i], i + offset * 3 / 2),
        xytext=(market_scores_furniture[i] + market_bias_furniture[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="dotted", color="yellowgreen"),
    )
    ax1.scatter(
        [market_scores_furniture[i], market_scores_furniture[i] + market_bias_furniture[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{market_bias_furniture[i]:.2f}",
        (market_scores_furniture[i] + market_bias_furniture[i], i + offset * 1.75),
        color="yellowgreen",
        ha="left",
        va="center",
    )

    # Usability bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(usability_scores_furniture[i], i + offset / 2),
        xytext=(usability_scores_furniture[i] + usability_bias_furniture[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="dodgerblue"),
    )
    ax1.scatter(
        [usability_scores_furniture[i], usability_scores_furniture[i] + usability_bias_furniture[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{usability_bias_furniture[i]:.2f}",
        (usability_scores_furniture[i] + usability_bias_furniture[i], i + offset * 0.75),
        color="dodgerblue",
        ha="right",
        va="center",
    )

# Second subplot (appliance brands)
for i, brand in enumerate(appliance_brands):
    ax2.annotate(
        "",
        xy=(market_scores_appliance[i], i + offset * 3 / 2),
        xytext=(market_scores_appliance[i] + market_bias_appliance[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="dotted", color="yellowgreen"),
    )
    ax2.scatter(
        [market_scores_appliance[i], market_scores_appliance[i] + market_bias_appliance[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{market_bias_appliance[i]:.2f}",
        (market_scores_appliance[i] + market_bias_appliance[i], i + offset * 1.75),
        color="yellowgreen",
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(usability_scores_appliance[i], i + offset / 2),
        xytext=(usability_scores_appliance[i] + usability_bias_appliance[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="dodgerblue"),
    )
    ax2.scatter(
        [usability_scores_appliance[i], usability_scores_appliance[i] + usability_bias_appliance[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{usability_bias_appliance[i]:.2f}",
        (usability_scores_appliance[i] + usability_bias_appliance[i], i + offset * 0.75),
        color="dodgerblue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(furniture_brands))
ax2.set_ylim(0, len(appliance_brands))

# Set x-axis limits uniformly
ax1.set_xlim(2, 14)
ax2.set_xlim(2, 14)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(furniture_brands))])
ax1.set_yticklabels(furniture_brands)
ax2.set_yticks([i + offset for i in range(len(appliance_brands))])
ax2.set_yticklabels(appliance_brands)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(furniture_brands))], minor=True)
ax2.set_yticks([i for i in range(len(appliance_brands))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
red_arrow = mlines.Line2D(
    [],
    [],
    color="yellowgreen",
    marker=">",
    linestyle="dotted",
    markersize=8,
    label=furniture_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="dodgerblue",
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=furniture_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[red_arrow, blue_arrow], bbox_to_anchor=(0.45, 0), ncol=2)

red_arrow = mlines.Line2D(
    [],
    [],
    color="yellowgreen",
    marker=">",
    linestyle="dotted",
    markersize=8,
    label=appliance_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="dodgerblue",
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=appliance_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[red_arrow, blue_arrow], bbox_to_anchor=(0.85, 0), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig('quiver_8.pdf', bbox_inches='tight')