import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data points for energy sources, their costs, and CO2 emissions.
energy_sources = ["Solar", "Wind", "Hydro", "Coal", "Nuclear", "Natural Gas"]
costs = [1.0, 1.5, 2.0, 1.5, 6.0, 4.0]  # Costs (in cents per kWh)
co2_emissions = [0.02, 0.03, 0.01, 1.0, 0.05, 1.0]  # CO2 emissions (kg CO2 per kWh)
colors = ["yellow", "blue", "green", "black", "purple", "orange"]
xlabel = "Cost (cents per kWh)"  # X-axis label for energy costs.
ylabel = "CO2 Emissions (kg CO2 per kWh)"  # Y-axis label for CO2 emissions.
title = "Energy Source Costs vs. CO2 Emissions"  # Title of the plot.
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and a subplot with a specific size.
fig, ax = plt.subplots(figsize=(6, 3))

# Scatter plot each energy source's data point and add text annotation.
for i in range(len(energy_sources)):
    ax.scatter(
        costs[i], co2_emissions[i], color=colors[i]
    )  # Plot data points with specific color.
    # Align text annotations based on position for better readability.
    ax.text(
        costs[i] + 0.1, co2_emissions[i] + 0.01, energy_sources[i], fontsize=10, ha="left", va="bottom"
    )

# Add a horizontal and a vertical dashed reference line.
ax.axhline(
    y=0.02, color="gray", linestyle="--", linewidth=1
)  # Horizontal line at CO2 Emissions=0.02 kg CO2 per kWh
ax.axvline(
    x=1.0, color="gray", linestyle="--", linewidth=1
)  # Vertical line at Cost=1.0 cents per kWh

# Set the scales of the axes to logarithmic.
ax.set_xscale("log")
ax.set_yscale("log")

# Set the labels for the axes.
ax.set_xlabel(xlabel)  # X-axis label for energy costs.
ax.set_ylabel(ylabel)  # Y-axis label for CO2 emissions.

# Set the title of the plot.
ax.set_title(title)  # Title of the plot.

# Adjust the tick labels to match the reference image.
# Define major ticks for both axes.
ax.set_xticks([0.5, 1, 1.5, 2, 3, 6, 10])
ax.get_xaxis().set_major_formatter(
    plt.ScalarFormatter()
)  # Format the tick labels as scalar values.
ax.set_yticks([0.01, 0.02, 0.03, 0.1, 1, 2.5])
ax.get_yaxis().set_major_formatter(
    plt.ScalarFormatter()
)  # Format the tick labels as scalar values.

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with a tight layout to ensure all elements fit within the figure area.
plt.tight_layout()
plt.savefig('scatter_7.pdf', bbox_inches='tight')
