# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np; np.random.seed(0)
# Generate sample data for health metrics
data_without_upgrades = np.random.beta(a=[5, 4, 3], b=[3, 4, 2], size=(100, 3))
data_with_upgrades = np.random.beta(a=[3, 5, 4], b=[2, 3, 4], size=(100, 3))
x_values = np.linspace(0, 1, 200)

# Define the categories and the colors for each group
categories = ["Internet Usage", "Device Connectivity", "Data Transfer Speed"]
title = "Technology Adoption Metrics with and without Infrastructure Upgrades"
offsets = [-0.05, 0, 0.05]
ylabel = "Percentage of Total Network Efficiency (%)"
labels = ["Without Upgrades", "With Upgrades"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 6))
colors = ["salmon", "limegreen"] 

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.015

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]
    # Plot data without upgrades
    kde_without_upgrades = gaussian_kde(data_without_upgrades[:, i])
    kde_without_upgrades_y = kde_without_upgrades(x_values)
    kde_without_upgrades_y_scaled = kde_without_upgrades_y / max(kde_without_upgrades_y) * violin_width
    ax.fill_betweenx(
        x_values,
        kde_without_upgrades_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
    )

    # Plot data with upgrades
    kde_with_upgrades = gaussian_kde(data_with_upgrades[:, i])
    kde_with_upgrades_y = kde_with_upgrades(x_values)
    kde_with_upgrades_y_scaled = kde_with_upgrades_y / max(kde_with_upgrades_y) * violin_width
    ax.fill_betweenx(
        x_values,
        offset,
        -kde_with_upgrades_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
    )

    # Plot the mean as a star marker for data without upgrades
    ax.plot(
        offset,
        np.mean(data_without_upgrades[:, i]),
        "*",
        color="white",
        markersize=12,
        markeredgecolor="black",
    )
    # Plot the mean as a star marker for data with upgrades
    ax.plot(
        offset,
        np.mean(data_with_upgrades[:, i]),
        "*",
        color="white",
        markersize=12,
        markeredgecolor="black",
    )

    ax.text(
        offset, -0.1, category, ha="center", va="top", fontsize=9
    )  # Add the category label below the violin plot

# Add title
ax.set_title(title, fontsize=14)

# Add grid
ax.grid(True, linestyle='--', alpha=0.7)

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.06,
    max(offsets) + scaling_factor * violin_width + 0.06,
)
y_margin = 0.02  # Adding margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black") for color in colors
]

ax.legend(handles, labels, loc="lower center", bbox_to_anchor=(0.5, -0.15), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the figure
plt.tight_layout()
plt.savefig('violin_11.pdf', bbox_inches='tight')