# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde



# ===================
# Part 2: Data Preparation
# ===================
import numpy as np; np.random.seed(0)
# Simulated power consumption data for Solar and Wind Grids across five seasons
solar_grid_data = np.random.beta(a=[4, 3, 2, 3, 2], b=[3, 4, 5, 3, 2], size=(100, 5))
wind_grid_data = np.random.beta(a=[5, 4, 3, 2, 3], b=[2, 3, 4, 5, 3], size=(100, 5))
ylabel = "Power Consumption (MW)"
ylim = [0, 1]
violin_width = 0.4
scaling_factor = 1
kde_x = np.linspace(0, 1, 300)

# Offsets for seasons
offsets_5_seasons = np.linspace(-3, 3, 5)
labels = ["Solar Grid", "Wind Grid"]
title = "Seasonal Power Consumption Distribution by Solar and Wind Grids"
legend_labels = ["Solar Grid", "Wind Grid"]
xticklabels = ["Early Spring", "Mid Summer", "Late Fall", "Early Winter", "Annual Average"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure
fig, ax = plt.subplots(figsize=(10, 6))

# Define the colors for each season
colors = ["#e41a1c", "#377eb8", "#4daf4a", "#984ea3", "#ff7f00"]

# Function to plot half violins for power consumption data
def plot_half_violins(ax, solar_data, wind_data, offsets, colors, labels, title, xticklabels):
    # Plot the half-violins with an offset for seasons
    for i in range(solar_data.shape[1]):
        offset = offsets[i]

        # Plot data for Solar Grid
        kde_solar = gaussian_kde(solar_data[:, i])
        kde_x = np.linspace(0, 1, 300)
        kde_solar_y = kde_solar(kde_x)
        kde_solar_y_scaled = kde_solar_y / max(kde_solar_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_solar_y_scaled * scaling_factor + offset,
            offset,
            color=colors[i],
            edgecolor="black",
            alpha=0.6
        )

        # Plot data for Wind Grid
        kde_wind = gaussian_kde(wind_data[:, i])
        kde_wind_y = kde_wind(kde_x)
        kde_wind_y_scaled = kde_wind_y / max(kde_wind_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_wind_y_scaled * scaling_factor + offset,
            color=colors[i],
            edgecolor="black",
            alpha=0.3
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for seasons
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)  # Set y-axis limits to 0-1 for beta distribution
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)  # Set x-ticks to the center of each season
    ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
    ax.title.set_text(title)

# Plot the violins
plot_half_violins(
    ax,
    solar_grid_data,
    wind_grid_data,
    offsets_5_seasons,
    colors,
    labels,
    title,
    xticklabels
)

# Add a legend for the entire figure
handles = [
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="black", markersize=10, alpha=0.6
    ),
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="black", markersize=10, alpha=0.3
    ),
]

fig.legend(handles, legend_labels, loc="lower center", bbox_to_anchor=(0.5, -0.1), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()

# Display the plot
plt.savefig('violin_16.pdf', bbox_inches='tight')