# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np; np.random.seed(0)
# Simulate new data representing technology performance metrics
nn_data = np.clip(np.random.normal(0.70, 0.08, 200), 0, 1)  # Neural Network
dt_data = np.clip(np.random.normal(0.60, 0.10, 200), 0, 1)  # Decision Tree
rf_data = np.clip(np.random.normal(0.68, 0.13, 200), 0, 1)  # Random Forest
svm_data = np.clip(np.random.normal(0.62, 0.12, 200), 0, 1)  # Support Vector Machine
knn_data = np.clip(np.random.normal(0.78, 0.10, 200), 0, 1)  # K-Nearest Neighbors

# Simulated metrics for Pearson R and Error Rate (EER)
pearson_r = [0.28, 0.24, 0.22, 0.19, 0.16]
eer = [4.0, 6.0, 8.5, 12.0, 18.0]

data = [nn_data, dt_data, rf_data, svm_data, knn_data]
categories = ["Neural Network", "Decision Tree", "Random Forest", "SVM", "KNN"]
ylabel = "Model Performance Score"
ylim = [0, 1.06]
xlabel = "Algorithm Type"
textlabels = ["Pearson Correlation", "Error Rate (%)"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)

# Define a technology-oriented color palette
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd"]

for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(0.75)

    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="w", s=40, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(i + 1 + 0.3, np.median(data[i]), f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with Pearson R and EER values
    ax.text(i + 1, 0.14, f"{pearson_r[i]:.2f}", ha="center", va="center", color="green", fontsize=10)
    ax.text(i + 1, 0.08, f"{eer[i]:.2f}", ha="center", va="center", color="blue", fontsize=10)

ax.text(5.6, 0.14, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, 0.08, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# Add grid for better readability
ax.grid(True, linestyle='--', which='both', color='grey', alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig('violin_20.pdf', bbox_inches='tight')