
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Fixing random state for reproducibility
np.random.seed(0)

# Domain: Monthly Average Temperatures over a year
months = np.array([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12])  # represents Jan to Dec
morning_temps = np.random.uniform(10, 25, 12)  # Random average morning temperatures
afternoon_temps = np.random.uniform(15, 30, 12)  # Random average afternoon temperatures
evening_temps = np.random.uniform(5, 20, 12)  # Random average evening temperatures

# Creating a 3x12 matrix with morning, afternoon, and evening temperatures
temperatures = np.vstack([morning_temps, afternoon_temps, evening_temps])

xedges = months
yedges = np.array([0, 1, 2, 3])  # Labels for Morning, Afternoon, Evening

xlabel = 'Month'
ylabel = 'Time of Day'
zlabel = 'Temperature (°C)'
title = 'Monthly Average Temperatures'
colorbar_label = 'Temperature (°C)'

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

# Construct arrays for the anchor positions of the bars.
xpos, ypos = np.meshgrid(xedges[:-1] + 0.5, np.array([0.5, 1.5, 2.5]), indexing="ij")
xpos = xpos.ravel()
ypos = ypos.ravel()
zpos = np.zeros_like(xpos)

# Construct arrays with the dimensions for the bars.
dx = dy = 0.8 * np.ones_like(zpos)
dz = temperatures.ravel()

fig = plt.figure(figsize=(10, 6))
ax = fig.add_subplot(projection="3d")

# Create a colormap for the color bar
colors = plt.cm.coolwarm(dz / 30)
bar3d = ax.bar3d(xpos, ypos, zpos, dx, dy, dz, zsort="average", color=colors)

# Add color bar which maps values to colors.
sm = plt.cm.ScalarMappable(cmap=plt.cm.coolwarm, norm=plt.Normalize(vmin=dz.min(), vmax=dz.max()))
sm.set_array([])
fig.colorbar(sm, ax=ax, shrink=0.5, aspect=10, label=colorbar_label)  # Adjust colorbar position and size

# Set labels and title
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_zlabel(zlabel)
ax.set_title(title)

# Set x and y ticks
ax.set_xticks(xedges[:-1] + 0.5)
ax.set_yticks([0.5, 1.5, 2.5])
ax.set_xticklabels(['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'])
ax.set_yticklabels(['Morning', 'Afternoon', 'Evening'])

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("3d_39.pdf", bbox_inches="tight")
