
# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt

from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)


# Fixing random state for reproducibility
def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]


x = np.linspace(0.0, 12.0, 41)
genre_numbers = range(1, 4)

# Example data: frequency of genres being read in different age groups
def genre_data(v, x):
    return np.sin(v * x / 6.0) + 1.5

# verts[i] is a list of (x, y) pairs defining polygon i.
verts = [
    polygon_under_graph(x, genre_data(v, x))
    for v in genre_numbers
]

xlabel = "Age Group"
ylabel = "Genre"
zlabel = "Reading Frequency"

# Adjust labels
genres = ['Fantasy', 'Mystery', 'Historical']

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
ax = plt.figure(figsize=(10, 8)).add_subplot(projection="3d")
facecolors = ['#8c564b', '#e377c2', '#7f7f7f']

poly = PolyCollection(verts, facecolors=facecolors, alpha=0.8)
ax.add_collection3d(poly, zs=genre_numbers, zdir="y")

ax.set(
    xlim=(0, 12),
    ylim=(1, 4),
    zlim=(0, 2.5),
    xlabel=xlabel,
    ylabel=ylabel,
    zlabel=zlabel,
)

# Set y-axis ticks to genre names
ax.set_yticks(genre_numbers)
ax.set_yticklabels(genres)
ax.set_box_aspect(aspect=(1,1.2,1.2), zoom=1)

# Title and annotations
plt.title("Reading Frequency of Different Genres Across Age Groups", pad=10)
ax.text(11, 3.5, 2.4, "Higher frequency", style='italic', fontsize=10, color='gray')
ax.text(11, 3.5, 0.1, "Lower frequency", style='italic', fontsize=10, color='gray')

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("3d_60.pdf", bbox_inches="tight")
