
# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
# Import numpy
import numpy as np

# Set a random seed for reproducibility
np.random.seed(0)


def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]


# Define the x-axis data
x = np.linspace(0.0, 10.0, 100)  # Increased resolution
fertilizer_levels = range(1, 4)


# New function to simulate crop yield over varying irrigation and fertilizer levels
def crop_yield(x, f):
    # simulate a quadratic response to irrigation, modulated by fertilizer level
    return (0.1 * x - 0.005 * x**2) * f


# Generate vertices for polygons
verts = [polygon_under_graph(x, crop_yield(x, f)) for f in fertilizer_levels]

# Extracted variables
x_label = "Irrigation (mm)"
y_label = "Fertilizer Application Level"
z_label = "Crop Yield (tons/ha)"
title = "Crop Yield by Irrigation and Fertilizer Levels"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and 3D axis
yticks = [1, 2, 3]
xlim = (0, 10)
ylim = (1, 4)
zlim = (0, 2.5)
aspect_ratio = [2, 1, 1]
view_elev = 30
view_azim = 150

fig = plt.figure(figsize=(10, 8))
ax = fig.add_subplot(projection="3d")

# Define the face colors
facecolors = ["#8FBC8F", "#556B2F", "#8B4513"]  # Colors reflecting agricultural theme

# Create a PolyCollection object
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=fertilizer_levels, zdir="y")

# Set the axis labels and limits
ax.set(xlim=xlim, ylim=ylim, zlim=zlim, xlabel=x_label, ylabel=y_label, zlabel=z_label)

# Add ticks and title
ax.set_yticks(yticks)
ax.set_title(title, pad=20)

# Adjust the aspect ratio and view angle
ax.set_box_aspect(aspect_ratio)
ax.view_init(elev=view_elev, azim=view_azim)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.savefig("3d_79.pdf", bbox_inches="tight")
