
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate random data to simulate the plot
data_algo1 = np.random.normal(1.5, 0.5, 100)
data_algo2 = np.random.normal(2.5, 0.3, 100)

# Combine data into a list
data = [data_algo1, data_algo2]

# Create positions for each box plot
positions = [0, 1]
xticks = ["Algorithm 1", "Algorithm 2"]
xlabel = "Design Algorithms"
ylabel = "Processing Time (Seconds)"
title = "Comparison of Design Algorithms"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size to match the original image's dimensions
plt.figure(figsize=(8, 6))

# Define new colors
colors = ["royalblue", "salmon"]
for i, d in enumerate(data):
    # Calculate KDE
    kde = gaussian_kde(d)
    kde_x = np.linspace(min(d), max(d), 300)
    kde_y = kde(kde_x)
    kde_y_scaled = kde_y / kde_y.max() * 0.2  # Adjusted scale factor
    offset = 0.25

    # Plot filled density plot to the left of the boxplot
    plt.fill_betweenx(
        kde_x,
        positions[i] - kde_y_scaled - offset,
        positions[i] - offset,
        alpha=0.5,
        color=colors[i],
        edgecolor="black",
    )

# Create box plots inside the violin plots
for i, d in enumerate(data):
    plt.boxplot(
        d,
        positions=[positions[i]],
        widths=0.1,
        patch_artist=True,
        medianprops=dict(color="black"),
        boxprops=dict(facecolor=colors[i], color="black"),
        capprops=dict(color="black"),
        whiskerprops=dict(color="black"),
    )

# Add scatter plot for individual data points
for i, d in enumerate(data):
    x = np.random.normal(positions[i], 0.04, size=len(d))
    plt.scatter(
        x,
        d,
        alpha=0.6,
        color=colors[i],
        edgecolor="black",
        s=50,  # Adjusted marker size for better visibility
    )

# Set the x-axis labels and add title
plt.xticks([0, 1], xticks)
plt.xlabel(xlabel)
plt.title(title)
plt.legend(xticks, loc="upper right")

# Set the y-axis label
plt.ylabel(ylabel)

# Adjust the y-axis limits
plt.ylim(0, 4)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_106.pdf", bbox_inches="tight")
