
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new data simulating income distributions for Engineers and Teachers
data_engineers = np.random.lognormal(mean=10, sigma=0.5, size=100)
data_teachers = np.random.lognormal(mean=9, sigma=0.5, size=100)

# Combine data into a list
data = [data_engineers, data_teachers]

# Create positions for each box plot
positions = [0, 1]
xticks = ["Engineers", "Teachers"]
xlabel = "Professions"
ylabel = "Income Distribution (Log Scale)"
title = "Income Distribution Among Different Professions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(10, 6))

# Define colors for the plots
colors = ["steelblue", "darkturquoise"]

# Calculate the kernel density estimate and plot the density plot for each dataset
for i, d in enumerate(data):
    # Calculate KDE
    kde = gaussian_kde(d)
    # Create an array of values covering the entire range of data for KDE
    kde_x = np.logspace(np.log10(min(d)), np.log10(max(d)), 300)
    kde_y = kde(kde_x)
    # Scale KDE values to match the position of the boxplot
    kde_y_scaled = kde_y / kde_y.max() * 0.1  # Scale factor can be adjusted
    # Plot filled density plot to the left of the boxplot
    offset = 0.2
    plt.fill_betweenx(
        kde_x,
        positions[i] - kde_y_scaled - offset,
        positions[i] - offset,
        alpha=0.5,
        color=colors[i],
        edgecolor="black",
    )

# Create box plots inside the violin plots
for i, d in enumerate(data):
    plt.boxplot(
        d,
        positions=[positions[i]],
        widths=0.05,
        patch_artist=True,
        medianprops=dict(color="black"),
        boxprops=dict(facecolor="none", color="black"),
    )

# Add scatter plot for individual data points
for i, d in enumerate(data):
    x = np.random.normal(positions[i], 0.04, size=len(d))
    plt.scatter(
        x,
        d,
        alpha=0.6,
        color=colors[i],
        edgecolors="w",
        s=20,
    )

# Set the x-axis labels and add title
plt.xticks([0, 1], xticks)
plt.xlabel(xlabel)
plt.title(title)

# Set the y-axis label
plt.ylabel(ylabel)

# Adjust the y-axis scale to logarithmic scale
plt.yscale('log')

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_107.pdf", bbox_inches="tight")
