
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
from scipy.stats import gaussian_kde

# Generating sample data representing agricultural yield in three different periods
yield_early = np.random.normal(1000, 50, 100)  # Early 20th century trend
yield_mid = np.random.normal(1500, 70, 100)  # Mid 20th century trend
yield_recent = np.random.normal(2000, 90, 100)  # Turn of the century trend

# Creating KDE for each sample data set to estimate the density of data points
kde_early = gaussian_kde(yield_early)
kde_mid = gaussian_kde(yield_mid)
kde_recent = gaussian_kde(yield_recent)

yield_range = np.linspace(800, 2200, 50)  # Defining the range of yields for plotting
labels = ["Early 20th Century", "Mid 20th Century", "Turn of the Century"]
ax1_legend_title = "Era"
ax1_xlabel = "Density"
ax2_ylabel = "Yield (kg/ha)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Setting up the figure and axes for a 1 x 2 layout
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))

# First subplot: Fill between plot along y-axis (KDE Plot)
ax1.fill_betweenx(yield_range, kde_early(yield_range), color="lightgreen", alpha=0.5)
ax1.fill_betweenx(yield_range, kde_mid(yield_range), color="saddlebrown", alpha=0.5)
ax1.fill_betweenx(yield_range, kde_recent(yield_range), color="forestgreen", alpha=0.5)
ax1.plot(kde_early(yield_range), yield_range, label=labels[0], color="green", linestyle='--', linewidth=2)
ax1.plot(kde_mid(yield_range), yield_range, label=labels[1], color="brown", linestyle='-.', linewidth=2)
ax1.plot(kde_recent(yield_range), yield_range, label=labels[2], color="darkgreen", linestyle='-', linewidth=2)
ax1.legend(title=ax1_legend_title, loc="upper right")
ax1.set_xlabel(ax1_xlabel)
ax1.set_yticks([])  # Hiding y-axis ticks for clarity

# Second subplot: Box plot for the same datasets along y-axis
box = ax2.boxplot(
    [yield_early, yield_mid, yield_recent], vert=True, patch_artist=True, medianprops={"color": "black"}
)
colors = ["lightgreen", "saddlebrown", "forestgreen"]  # Color matching with KDE plot
for patch, color in zip(box["boxes"], colors):
    patch.set_facecolor(color)

ax2.set_ylabel(ax2_ylabel)
ax2.set_xticks([])  # Hiding x-axis ticks for clarity
ax2.set_ylim(800, 2200)  # Setting limits for y-axis to align with the KDE plot

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_114.pdf", bbox_inches="tight")
