
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import norm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulating star luminosity data for Red Giants and White Dwarfs
luminosity_red_giants = np.random.normal(loc=5, scale=1.5, size=500)
luminosity_white_dwarfs = np.random.normal(loc=2, scale=0.5, size=500)

# Labels and Plot Types
ax1_label = "Red Giants"
ax2_label = "White Dwarfs"
ax1_text = r"$\mathrm{Red\ Giants}$"
ax2_text = r"$\mathrm{White\ Dwarfs}$"

# Axes Limits and Labels
xlabel_value = "Luminosity"
ylabel_value = "Density"
xticks_values = [-1, 0, 1, 2, 3, 4, 5, 6, 7, 8]
xlim_values = [-2, 9]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(
    figsize=(10, 8)
)

# Plot histograms
ax.hist(luminosity_red_giants, bins=15, density=True, alpha=0.75, color="#FF6347", label=ax1_label)
ax.hist(luminosity_white_dwarfs, bins=15, density=True, alpha=0.75, color="#4682B4", label=ax2_label)

# Plot normal distributions
xmin, xmax = ax.get_xlim()
x = np.linspace(xmin, xmax, 100)
p_red_giants = norm.pdf(x, np.mean(luminosity_red_giants), np.std(luminosity_red_giants))
p_white_dwarfs = norm.pdf(x, np.mean(luminosity_white_dwarfs), np.std(luminosity_white_dwarfs))
ax.plot(x, p_red_giants, color="#FF4500", linewidth=3, linestyle='solid')
ax.plot(x, p_white_dwarfs, color="#1E90FF", linewidth=3, linestyle='solid')

# Add dashed lines at mean
ax.axvline(np.mean(luminosity_red_giants), color="black", linestyle="dashed")
ax.axvline(np.mean(luminosity_white_dwarfs), color="black", linestyle="dashed")

# Add text labels for dashed lines
ax.text(
    np.mean(luminosity_red_giants) + 0.1,
    ax.get_ylim()[1] - 0.01,
    ax1_text,
    ha="center",
    va="top",
    fontsize=18,
)
ax.text(
    np.mean(luminosity_white_dwarfs) + 0.1,
    ax.get_ylim()[1] - 0.005,
    ax2_text,
    ha="center",
    va="top",
    fontsize=18,
)

# Set labels and title
ax.set_xlabel(xlabel_value, fontsize=14, labelpad=15)
ax.set_ylabel(ylabel_value, fontsize=14, labelpad=15)
ax.set_xticks(xticks_values)
ax.set_xlim(xlim_values)
ax.set_title("Distribution of Star Luminosities", fontsize=16, pad=20)

# Add legend
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("CB_32.pdf", bbox_inches="tight")
