
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate some health-related data
# BMI data
bmi = np.random.normal(25, 5, 100)

# Simulate blood pressure readings with a linear relationship to BMI
blood_pressure = 0.9 * bmi + np.random.normal(0, 5, 100)

# Trend line data
trend_x = [15, 20, 25, 30, 35]
trend_y = [13.5, 18, 22.5, 27, 31.5]
diagonal_line = [[15, 35], [13.5, 31.5]]

scatters_name = "BMI vs Blood Pressure"
bin_edges_name = "Dot edges"
calibration_curve_name = "Trend line"

line_label = "Trend line"
xlabel = "BMI"
ylabel = "Blood Pressure"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Draw the scatter plot and marginals.

# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# no labels
ax_histx.tick_params(axis="x", labelbottom=False)

# the scatter plot:
# add different colors to the scatter plot
sc = ax.scatter(bmi, blood_pressure, c=bmi, cmap="winter", s=50, edgecolor="black", alpha=0.7)

# # add a line on ax
ax.plot(
    trend_x,
    trend_y,
    color="seagreen",
    linewidth=2,
    marker="o",
    markersize=6,
    label=line_label,
)

# draw a diagonal line
ax.plot(diagonal_line[0], diagonal_line[1], color="gray", linestyle="--")

# now determine nice limits by hand:
binwidth = 1
xymax = max(np.max(np.abs(bmi)), np.max(np.abs(blood_pressure)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(bmi, bins=bins, color="#76a5af", edgecolor="black")

# remove the y-axis labels
ax_histx.set_yticks([])

# remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

ax.set_xticks(np.arange(15, 36, 5))
ax.set_yticks(np.arange(13.5, 32, 4.5))
ax.set_xlim(10, 40)
ax.set_ylim(10, 35)

# draw bin edges
for i in bins:
    ax.axvline(i, color="gray", linewidth=0.5, linestyle="--", zorder=0, alpha=0.5)

# Create an axes on the top side of ax_heatmap_top for the colorbar.
ax_colorbar = fig.add_axes(
    [ax.get_position().width * 1.15, 0.1, 0.05, ax.get_position().y1 * 0.86]
)
cbar = plt.colorbar(sc, cax=ax_colorbar, orientation="vertical")

# Create a Line2D instance for bin edges
bin_edge_line = Line2D([0], [0], color="gray", linewidth=0.5, linestyle="--")

# Create a Line2D instance for trend line
calibration_curve_line = Line2D(
    [0],
    [0],
    color="seagreen",
    linewidth=2,
    marker="o",
    markersize=6,
)

# Update legend to include the new Line2D instances
ax.legend(
    handles=[sc, bin_edge_line, calibration_curve_line],
    labels=[scatters_name, bin_edges_name, calibration_curve_name],
    loc="upper left",
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_37.pdf", bbox_inches="tight")
