
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Define parameters for three product categories
distributions = {
    "Electronics": {"mean": [50, 2000], "cov": [[100, 1500], [1500, 25000]]},
    "Clothing": {"mean": [30, 1500], "cov": [[80, 1200], [1200, 20000]]},
    "Groceries": {"mean": [20, 1000], "cov": [[60, 1000], [1000, 15000]]},
}
keys = ["Electronics", "Clothing", "Groceries"]
# Generate samples
samples = {
    species: np.random.multivariate_normal(dist["mean"], dist["cov"], 100)
    for species, dist in distributions.items()
}

xlabel = "Sales Volume"
ylabel = "Marketing Spend"
title = "Product Categories"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the plotting grid
fig = plt.figure(figsize=(10, 8))
grid = plt.GridSpec(4, 4, hspace=0, wspace=0)

# Main scatter plot
main_ax = fig.add_subplot(grid[1:, :-1])
colors = {keys[0]: "teal", keys[1]: "orange", keys[2]: "purple"}
markers = {keys[0]: "o", keys[1]: "^", keys[2]: "s"}
for species, color in colors.items():
    subset = samples[species]
    main_ax.scatter(subset[:, 0], subset[:, 1], c=color, label=species, marker=markers[species], alpha=0.6)
main_ax.set_xlabel(xlabel)
main_ax.set_ylabel(ylabel)

# Top density plot
top_ax = fig.add_subplot(grid[0, :-1], sharex=main_ax)
all_samples = np.concatenate([samples[species] for species in samples], axis=0)
x_min, x_max = all_samples[:, 0].min(), all_samples[:, 0].max()
xs = np.linspace(x_min, x_max, 200)
for species, color in colors.items():
    density = gaussian_kde(samples[species][:, 0])
    top_ax.plot(xs, density(xs), linestyle='--', alpha=0.6, color=color)
    top_ax.fill_between(xs, density(xs), alpha=0.4, color=color)

# Right density plot
right_ax = fig.add_subplot(grid[1:, -1], sharey=main_ax)
y_min, y_max = all_samples[:, 1].min(), all_samples[:, 1].max()
ys = np.linspace(y_min, y_max, 200)
for species, color in colors.items():
    density = gaussian_kde(samples[species][:, 1])
    right_ax.plot(density(ys), ys, linestyle='--', alpha=0.6, color=color)
    right_ax.fill_betweenx(ys, density(ys), alpha=0.4, color=color)

# Hide the spines
top_ax.spines["top"].set_visible(False)
top_ax.spines["right"].set_visible(False)
top_ax.spines["left"].set_visible(False)
right_ax.spines["top"].set_visible(False)
right_ax.spines["right"].set_visible(False)
right_ax.spines["bottom"].set_visible(False)

# Remove the labels from the top and right axes
top_ax.tick_params(axis="x", which="both", top=False, bottom=False, labelbottom=False)
right_ax.tick_params(axis="y", which="both", left=False, right=False, labelleft=False)

# Remove all ticks from the right axis
top_ax.set_yticks([])
right_ax.set_xticks([])

main_ax.legend(title=title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_48.pdf", bbox_inches="tight")
