
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import norm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Define the data for each subplot using random numbers and normal distribution
mobile_usage_data = np.random.normal(5.5, 1.2, 1000)  # Mobile usage in hours
desktop_usage_data = np.random.normal(3.8, 0.7, 1000)  # Desktop usage in hours
tablet_usage_data = np.random.normal(2.2, 0.5, 1000)  # Tablet usage in hours
smartwatch_usage_data = np.random.normal(1.1, 0.3, 1000)  # Smartwatch usage in hours
titles = ["Mobile Usage", "Desktop Usage", "Tablet Usage", "Smartwatch Usage"]
xlabels = [
    "Usage time (hours)",
    "Usage time (hours)",
    "Usage time (hours)",
    "Usage time (hours)",
]
ylabels = ["Probability", "Probability", "Probability", "Probability"]
super_title = "Daily Device Usage Statistics"
annotation_positions = {
    "Mobile Usage": (0.6, 0.8),
    "Desktop Usage": (0.6, 0.8),
    "Tablet Usage": (0.6, 0.8),
    "Smartwatch Usage": (0.6, 0.8),
}

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size to match the original image's dimensions
plt.figure(figsize=(12, 10))

# Define the subplot grid with reduced space between subplots
grid = plt.GridSpec(2, 2, wspace=0.3, hspace=0.4)

# Color mapping for each dataset
colors = {
    "Mobile Usage": "#ff9999",
    "Desktop Usage": "#66b3ff",
    "Tablet Usage": "#99ff99",
    "Smartwatch Usage": "#ffcc99"
}

# Mobile usage subplot
ax_mobile = plt.subplot(grid[0, 0])
ax_mobile.hist(
    mobile_usage_data, bins=20, color=colors["Mobile Usage"], edgecolor="black", density=True, linewidth=1
)
ax_mobile.plot(
    np.sort(mobile_usage_data),
    norm.pdf(np.sort(mobile_usage_data), np.mean(mobile_usage_data), np.std(mobile_usage_data)),
    color="#555555",
    linestyle='--'
)
ax_mobile.set_title(titles[0])
ax_mobile.set_xlabel(xlabels[0])
ax_mobile.set_ylabel(ylabels[0])
ax_mobile.set_ylim(0, 0.35)
ax_mobile.text(
    annotation_positions["Mobile Usage"][0],
    annotation_positions["Mobile Usage"][1],
    f"$\mu = {np.mean(mobile_usage_data):.2f}$\n$\sigma = {np.std(mobile_usage_data):.2f}$",
    transform=ax_mobile.transAxes,
    fontsize=9,
)

# Desktop usage subplot
ax_desktop = plt.subplot(grid[0, 1])
ax_desktop.hist(
    desktop_usage_data,
    bins=20,
    color=colors["Desktop Usage"],
    edgecolor="black",
    density=True,
    linewidth=1,
)
ax_desktop.plot(
    np.sort(desktop_usage_data),
    norm.pdf(np.sort(desktop_usage_data), np.mean(desktop_usage_data), np.std(desktop_usage_data)),
    color="#555555",
    linestyle='--'
)
ax_desktop.set_title(titles[1])
ax_desktop.set_xlabel(xlabels[1])
ax_desktop.set_ylabel(ylabels[1])
ax_desktop.set_ylim(0, 0.6)
ax_desktop.text(
    annotation_positions["Desktop Usage"][0],
    annotation_positions["Desktop Usage"][1],
    f"$\mu = {np.mean(desktop_usage_data):.2f}$\n$\sigma = {np.std(desktop_usage_data):.2f}$",
    transform=ax_desktop.transAxes,
    fontsize=9,
)

# Tablet usage subplot
ax_tablet = plt.subplot(grid[1, 0])
ax_tablet.hist(
    tablet_usage_data,
    bins=20,
    color=colors["Tablet Usage"],
    edgecolor="black",
    density=True,
    linewidth=1,
)
ax_tablet.plot(
    np.sort(tablet_usage_data),
    norm.pdf(np.sort(tablet_usage_data), np.mean(tablet_usage_data), np.std(tablet_usage_data)),
    color="#555555",
    linestyle='--'
)
ax_tablet.set_title(titles[2])
ax_tablet.set_xlabel(xlabels[2])
ax_tablet.set_ylabel(ylabels[2])
ax_tablet.set_ylim(0, 1.0)
ax_tablet.text(
    annotation_positions["Tablet Usage"][0],
    annotation_positions["Tablet Usage"][1],
    f"$\mu = {np.mean(tablet_usage_data):.2f}$\n$\sigma = {np.std(tablet_usage_data):.2f}$",
    transform=ax_tablet.transAxes,
    fontsize=9,
)

# Smartwatch usage subplot
ax_smartwatch = plt.subplot(grid[1, 1])
ax_smartwatch.hist(
    smartwatch_usage_data, bins=20, color=colors["Smartwatch Usage"], edgecolor="black", density=True, linewidth=1
)
ax_smartwatch.plot(
    np.sort(smartwatch_usage_data),
    norm.pdf(np.sort(smartwatch_usage_data), np.mean(smartwatch_usage_data), np.std(smartwatch_usage_data)),
    color="#555555",
    linestyle='--'
)
ax_smartwatch.set_title(titles[3])
ax_smartwatch.set_xlabel(xlabels[3])
ax_smartwatch.set_ylabel(ylabels[3])
ax_smartwatch.set_ylim(0, 2.0)
ax_smartwatch.text(
    annotation_positions["Smartwatch Usage"][0],
    annotation_positions["Smartwatch Usage"][1],
    f"$\mu = {np.mean(smartwatch_usage_data):.2f}$\n$\sigma = {np.std(smartwatch_usage_data):.2f}$",
    transform=ax_smartwatch.transAxes,
    fontsize=9,
)

# Add a super title to the figure for context
plt.suptitle(super_title, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout(rect=[0, 0.03, 1, 0.97])  # Adjust rect to accommodate the super title
plt.savefig("CB_95.pdf", bbox_inches="tight")
