
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib.colors as colors

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Function to draw ellipses with varying radii and colors
xlim = [-10, 10]
ylim = [-7, 7]
xlabel = "Distance (km)"
ylabel = "Time (hours)"
num_ellipses = 12
a_start = 3
b_start = 1
a_end = 9
b_end = 3
title = "Vehicle Movement Patterns"
color_labels = ["6AM", "7AM", "8AM", "9AM", "10AM", "11AM", "12PM", "1PM", "2PM", "3PM", "4PM", "5PM"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def draw_colored_ellipses(num_ellipses, a_start, b_start, a_end, b_end):
    t = np.linspace(0, 2 * np.pi, 100)
    fig, ax = plt.subplots(figsize=(8, 6))

    # Create a colormap
    cmap = cm.get_cmap("coolwarm", num_ellipses)
    norm = colors.BoundaryNorm(np.arange(num_ellipses + 1), cmap.N)

    for i in range(num_ellipses):
        # Interpolate the semi-major and semi-minor axes
        a = np.linspace(a_start, a_end, num_ellipses)[i]
        b = np.linspace(b_start, b_end, num_ellipses)[i]

        # Parametric equations for the ellipse
        x = a * np.cos(t)
        y = b * np.sin(t)

        # Use a colormap to determine the color
        color = cmap(i)
        ax.set_xlim(xlim)
        ax.set_ylim(ylim)
        ax.plot(x, y, linestyle='--', linewidth=1.5, color=color)
        ax.set_xlabel(xlabel)
        ax.set_ylabel(ylabel)

    # Add some annotations
    ax.annotate("Peak Traffic", xy=(0.5, 2.0), xytext=(2, 3),
                arrowprops=dict(facecolor='black', shrink=0.05))

    # Add a grid
    ax.grid(True, linestyle='--', linewidth=0.5)
    ax.set_aspect("equal")

    # Add Title
    ax.set_title(title)

    # Create a mappable object for the colorbar
    sm = cm.ScalarMappable(cmap=cmap, norm=norm)
    sm.set_array([])

    # Add the colorbar
    cbar = plt.colorbar(sm, ticks=np.arange(0.5, num_ellipses, 1), label="Time of Day", ax=ax)
    cbar.set_ticklabels(color_labels)  # set tick labels to specified times
    cbar.ax.tick_params(length=0)


# Show the plot
draw_colored_ellipses(
    num_ellipses=num_ellipses,
    a_start=a_start,
    b_start=b_start,
    a_end=a_end,
    b_end=b_end,
)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
plt.savefig("HR_29.pdf", bbox_inches="tight")
