
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Ellipse

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Generating data representing plant growth over time with error ellipses
time = np.linspace(0, 12, 400)  # Time from 0 to 12 months
growth_rate = 0.4 * (time - 6) ** 2 + 0.5  # Parabola for growth trend
error_magnitude = np.linspace(0, 2, len(time))

# Measurement points with some offset
measurement_time = np.linspace(2, 10, 20)
measurement_growth = 0.4 * (measurement_time - 6) ** 2 + 0.5 - 0.05

# Labels and Plot Types
label_trend = "Average Growth Rate"
label_error = "± 1-σ of Measurement"
label_measurement = "Individual Measurements"
ylim_values = [0, 16]
ellipse_sizes = np.linspace(0, 2, len(measurement_time))

# Axes Limits and Labels
xlabel_value = "Time (months)"
ylabel_value = "Growth Rate (cm/month)"
title = "Plant Growth Over Time with Measurement Error"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 6))
plt.plot(time, growth_rate, label=label_trend, color="#4CAF50", linestyle="--", linewidth=2)

for i, (px, py, size) in enumerate(zip(measurement_time, measurement_growth, ellipse_sizes)):
    if i == 0:  # Only for the first ellipse add label
        ellipse = Ellipse(
            (px, py),
            width=size,
            height=size / 2,
            edgecolor="#388E3C",
            facecolor="none",
            label=label_error,
        )
    else:
        ellipse = Ellipse(
            (px, py), width=size, height=size / 2, edgecolor="#388E3C", facecolor="none"
        )
    plt.gca().add_patch(ellipse)

plt.scatter(measurement_time, measurement_growth, color="#388E3C", label=label_measurement, zorder=5, marker='o')
plt.xlabel(xlabel_value)
plt.ylabel(ylabel_value)
plt.title(title)
plt.ylim(ylim_values)
plt.grid(True, color="white")
plt.gca().set_facecolor("#f4f4f4")
plt.gca().set_axisbelow(True)

plt.legend(facecolor="#f4f4f4")
for spine in plt.gca().spines.values():
    spine.set_visible(False)
plt.gca().tick_params(axis="both", length=0)  # Hide tick marks

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_35.pdf", bbox_inches="tight")
