
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generating synthetic data for GDP growth rates (in percentage) over 10 years
years = np.arange(2010, 2020)
gdp_growth_actual = np.array([2.5, 3.0, 2.8, 3.5, 2.7, 3.8, 2.9, 3.2, 2.4, 3.1])
gdp_growth_predicted = gdp_growth_actual + np.random.normal(0, 0.2, len(gdp_growth_actual))

# Error ellipses data (showing variance in prediction) reduced for simplicity
ellipse_sizes = np.linspace(0.5, 1.5, len(years))

# Labels and Plot Types
label_actual = "Actual GDP Growth"
label_predicted = "Predicted GDP Growth ± 1-σ"
label_points = "Observed Data"
ylim_values = [0, 5]

# Axes Limits and Labels
xlabel_value = "Year"
ylabel_value = "GDP Growth Rate (%)"
title = "GDP Growth Rate Over Time (2010-2019)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 6))
plt.plot(years, gdp_growth_actual, label=label_actual, color="#388e3c", linestyle='-', linewidth=2)
plt.plot(years, gdp_growth_predicted, label=label_predicted, color="#fbc02d", linestyle='--', linewidth=2)

# Plot error ellipses
for i, (year, growth, size) in enumerate(zip(years, gdp_growth_predicted, ellipse_sizes)):
    ellipse = plt.matplotlib.patches.Ellipse(
        (year, growth), width=0.3, height=size, edgecolor="#ff7043", facecolor="none"
    )
    plt.gca().add_patch(ellipse)

# Scatter plot of actual data points
plt.scatter(years, gdp_growth_actual, color="#303f9f", label=label_points, zorder=5, marker='o')

plt.xlabel(xlabel_value)
plt.ylabel(ylabel_value)
plt.title(title)
plt.ylim(ylim_values)
plt.grid(True, color="white")
plt.gca().set_facecolor("#f5f5f5")
plt.gca().set_axisbelow(True)

# Adjusting the legend
plt.legend(facecolor="#f5f5f5")
for spine in plt.gca().spines.values():
    spine.set_visible(False)
plt.gca().tick_params(axis="both", length=0)  # Hide tick marks

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_37.pdf", bbox_inches="tight")
