
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib import patches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Financial metrics and their impacts
features = [
    "Advertising Revenue",
    "Operational Costs",
    "Product Returns",
    "Customer Acquisition Cost",
    "Net Profit Margin",
    "Revenue Growth",
    "Market Share",
    "Operating Income",
    "Gross Profit Margin",
    "Sales",
    "EBITDA",
]

impact_values = [0.05, -0.03, -0.02, -0.04, 0.07, 0.06, 0.08, -0.01, 0.03, 0.10, 0.12]
start_x = 0.40
cumulative_values = [start_x]

axvhline = 0.70
ylim = [-1, 11]
xlim = [0.30, 0.80]
xticks = [0.35, 0.40, 0.45, 0.50, 0.55, 0.60, 0.65, 0.70, 0.75]
textposition = [[0.70, 10.5], [0.40, -0.5]]
textlabels = ["f(x) = 0.70", "f(x) = 0.40"]

xlabel = "Cumulative Impact on Key Financial Metrics"
ylabel = "Financial Metrics"
title = "Cumulative Impact of Financial Metrics on Business Performance"
supertitle = "Financial Performance Analysis"
legendlabels = ["Positive Impact", "Negative Impact"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
for impact in impact_values:
    next_value = cumulative_values[-1] + impact
    cumulative_values.append(next_value)

colors = ["#56B356" if impact > 0 else "#E04A42" for impact in impact_values]

fig, ax = plt.subplots(figsize=(10, 8))

for i in range(len(impact_values) - 1):
    plt.plot(
        [cumulative_values[i + 1], cumulative_values[i + 1]],
        [features[i], features[i + 1]],
        "k--",
        linewidth=0.5,
        color="gray",
    )

for idx, (feature, impact) in enumerate(zip(features, impact_values)):
    left = cumulative_values[idx]
    bar_width = 0.6

    if impact > 0:
        color = "#56B356"
        width = impact
        text_x = left + width
        ax.add_patch(
            patches.Rectangle(
                (left, idx - bar_width / 2),
                width - 0.005,
                bar_width,
                facecolor=color,
            )
        )
        ax.add_patch(
            patches.Polygon(
                [
                    (left + width - 0.005, idx - bar_width / 2),
                    (left + width, idx),
                    (left + width - 0.005, idx + bar_width / 2),
                ],
                closed=True,
                facecolor=color,
            )
        )
        ax.text(text_x, idx, f"+{impact}", va="center", ha="left", color="#56B356")
    else:
        color = "#E04A42"
        width = -impact
        text_x = left - width
        if impact == 0:
            ax.add_patch(
                patches.Rectangle(
                    (left - width, idx - bar_width / 2),
                    width,
                    bar_width,
                    facecolor=color,
                    edgecolor=color,
                )
            )
        else:
            ax.add_patch(
                patches.Rectangle(
                    (left - width + 0.005, idx - bar_width / 2),
                    width - 0.005,
                    bar_width,
                    facecolor=color,
                )
            )
            ax.add_patch(
                patches.Polygon(
                    [
                        (left - width + 0.005, idx - bar_width / 2),
                        (left - width, idx),
                        (left - width + 0.005, idx + bar_width / 2),
                    ],
                    closed=True,
                    facecolor=color,
                )
            )

        ax.text(
            text_x, idx, f"-{np.abs(impact)}", va="center", ha="right", color="#E04A42"
        )

ax.axvline(axvhline, linestyle="--", color="gray", alpha=0.3)
ax.yaxis.grid(True, linestyle="dotted", color="gray", alpha=0.3)
ax.set_axisbelow(True)
ax.tick_params(axis="both", length=0)
ax.spines["right"].set_visible(False)
ax.spines["top"].set_visible(False)
ax.spines["left"].set_visible(False)
ax.set_ylim(ylim)
ax.set_xlim(xlim)
ax.set_xticks(xticks)

ax.text(
    textposition[0][0],
    textposition[0][1],
    textlabels[0],
    va="bottom",
    ha="center",
    color="grey",
)
ax.text(
    textposition[1][0],
    textposition[1][1],
    textlabels[1],
    va="bottom",
    ha="center",
    color="grey",
)

ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
plt.title(title, fontsize=14)
fig.suptitle(supertitle, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("HR_60.pdf", bbox_inches="tight")
