
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Placeholder data for economic indicators
years = np.linspace(2000, 2020, 10)
gdp_growth = np.logspace(0, 1, 10)  # GDP growth rate
inflation = np.exp(-0.1 * years + 202)  # Inflation rate (decaying)
unemployment = np.exp(0.05 * years - 90)  # Unemployment rate (growing)

# Create error bands
gdp_growth_err = gdp_growth * 0.1
inflation_err = inflation * 0.15
unemployment_err = unemployment * 0.2

xlabel = "Year"
ylabel1 = "GDP Growth Rate (%)"
ylabel2 = "Rate (%)"
title1 = "GDP Growth Rate Over Years"
title2 = "Inflation and Unemployment Rates Over Years"
legendlabels = ["GDP Growth", "Inflation", "Unemployment"]
filename = "HR_15_9.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes with specified size (width, height)
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))

# Plot data and curves for the left subplot (GDP Growth)
ax1.plot(years, gdp_growth, "o", color="blue", label=legendlabels[0])
ax1.fill_between(
    years,
    gdp_growth - gdp_growth_err,
    gdp_growth + gdp_growth_err,
    color="blue",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(years, np.log(gdp_growth), 2)
fit_func = np.poly1d(coeffs)
ax1.plot(years, np.exp(fit_func(years)), "k--")

ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel1)
ax1.set_title(title1)
ax1.legend(frameon=True)

# Plot data and curves for the right subplot (Inflation and Unemployment)
ax2.plot(years, inflation, "s", color="green", label=legendlabels[1])
ax2.fill_between(
    years,
    inflation - inflation_err,
    inflation + inflation_err,
    color="green",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(years, np.log(inflation), 2)
fit_func = np.poly1d(coeffs)
ax2.plot(years, np.exp(fit_func(years)), "k--")

ax2.plot(years, unemployment, "^", color="red", label=legendlabels[2])
ax2.fill_between(
    years,
    unemployment - unemployment_err,
    unemployment + unemployment_err,
    color="red",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(years, np.log(unemployment), 2)
fit_func = np.poly1d(coeffs)
ax2.plot(years, np.exp(fit_func(years)), "k--")

ax2.set_xlabel(xlabel)
ax2.set_ylabel(ylabel2)
ax2.set_title(title2)
ax2.legend(frameon=True)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("HR_70.pdf", bbox_inches="tight")
