
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

number_of_bins = 40

# Defining three data sets for a psychology study
number_of_data_points = 387
labels = ["Low Stress Group", "Moderate Stress Group", "High Stress Group"]
data_sets = [
    np.random.normal(2, 1.5, number_of_data_points),  # Low stress levels
    np.random.normal(5, 2, number_of_data_points),   # Moderate stress levels
    np.random.normal(8, 1.2, number_of_data_points),  # High stress levels
]

# The bin_edges are the same for all of the histograms
hist_range = (np.min(data_sets), np.max(data_sets))
bin_edges = np.linspace(hist_range[0], hist_range[1], number_of_bins + 1)
centers = bin_edges[:-1] + np.diff(bin_edges) / 2
xlabel = "Frequency"
ylabel = "Stress Level"
title = "Distribution of Stress Levels in Different Groups"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Cycle through and plot each histogram
# Computed quantities to aid plotting
colors = ["#add8e6", "#ffb6c1", "#90ee90"]  # Light blue, pink, light green
binned_data_sets = [
    np.histogram(d, range=hist_range, bins=number_of_bins)[0] for d in data_sets
]
binned_maximums = np.max(binned_data_sets, axis=1)
x_locations = np.arange(0, sum(binned_maximums), np.max(binned_maximums))

fig, ax = plt.subplots(figsize=(10, 6))
for x_loc, binned_data, color in zip(x_locations, binned_data_sets, colors):
    lefts = x_loc - 0.5 * binned_data
    ax.barh(
        centers, binned_data, height=np.diff(bin_edges), left=lefts,
        color=color, edgecolor="black", alpha=0.7
    )

# Set the x-axis labels
ax.set_xticks(x_locations)
ax.set_xticklabels(labels)

# Set labels and title with specific names
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_89.pdf", bbox_inches="tight")
