
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
from matplotlib.ticker import FuncFormatter

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# New Dataset for Healthcare Domain
np.random.seed(42)
# Healthcare financial changes over a year
increments = [500, -200, 150, -250, 300, -100, 200, -150, 100, -50, 400]

# Labels for each stage
labels = [
    "Initial Funding",
    "Salaries",
    "Equipment",
    "Maintenance",
    "Research",
    "Utilities",
    "Medications",
    "Administration",
    "Grants",
    "Miscellaneous",
    "Final Balance",
]

# Determine starting point
start_value = 1000
end_value = start_value + sum(increments)

# Calculate the bottom of each bar (cumulative sums)
bottoms = np.hstack(([start_value], np.cumsum(increments)[:-1])) + start_value

# Axes Limits and Labels
ylabel_value = "Amount ($)"
title = "Healthcare Financial Overview"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the colors based on increment or decrement with a new color palette
colors = ["#77dd77" if x > 0 else "#ff6961" for x in increments]

fig, ax = plt.subplots(figsize=(10, 6))

# Plot bars
bars = ax.bar(labels, increments, bottom=bottoms, color=colors)

# Plot lines connecting the tops of each bar
for i in range(len(increments) - 1):  # Exclude the last increment
    start_top = bottoms[i] + increments[i]
    end_top = bottoms[i + 1] + increments[i + 1]
    ax.plot([i, i + 1], [start_top, end_top], color="skyblue", linestyle="-")

# Annotate bars with value labels
for i, bar in enumerate(bars):
    height = bar.get_height()
    ax.annotate(
        f"${height}",
        xy=(bar.get_x() + bar.get_width() / 2, bar.get_y() + height),
        xytext=(
            0,
            3 if height > 0 else -12,
        ),  # 3 points vertical offset or -12 if negative
        textcoords="offset points",
        ha="center",
        va="bottom",
        fontsize=10,
        bbox=dict(boxstyle="round,pad=0.3", edgecolor="black", facecolor="white"),
    )

# Set the y-axis label and title
ax.set_ylabel(ylabel_value)
ax.set_title(title, fontsize=14, fontweight='bold')

# Format y-axis as currency
formatter = FuncFormatter(lambda y, _: f"${int(y)}")
ax.yaxis.set_major_formatter(formatter)

# Remove x-axis line and ticks
ax.spines["bottom"].set_visible(False)
ax.xaxis.set_ticks([])

# Set grid
ax.grid(True, axis="y", linestyle="--", linewidth=0.7, alpha=0.7)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_94.pdf", bbox_inches="tight")
