
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generating new geographical data
clusters = {
    "city": np.random.normal(loc=(0, 2), scale=1, size=(50, 2)),
    "forest": np.random.normal(loc=(5, 5), scale=1, size=(30, 2)),
    "desert": np.random.normal(loc=(-3, 0), scale=1, size=(40, 2)),
    "mountain": np.random.normal(loc=(5, -5), scale=1, size=(60, 2)),
    "lake": np.random.normal(loc=(-5, -5), scale=1, size=(70, 2)),
    "agriculture": np.random.normal(loc=(5, 10), scale=1, size=(45, 2)),
}

insetaxes = [0.2, 0.6, 0.3, 0.2]    # Inset position
insetxlim = [-2, 0]                 # Inset x-axis limit
insetylim = [-0.5, 1.5]             # Inset y-axis limit
insetxticks = [-2.0, -1.0, 0]       # Inset x-axis ticks
insetyticks = [-0.5, 0.5, 1.5]      # Inset y-axis ticks
arrowstart = (-3.0, 4.0)            # Arrow start coordinates
arrowend = (0.38, 0.5)              # Arrow end coordinates
annotaterecx = [-2, 0]              # Annotation box x-coordinates
annotaterecy = [-0.5, 1.5]          # Annotation box y-coordinates
xlabel = "Longitude"                # x-axis label
ylabel = "Latitude"                 # y-axis label
title = "Geographical Data Clusters" # Plot title
inset_title = "Zoomed View"         # Inset title

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot

# Colors for each cluster reflecting geographical features
colors = {
    "city": "#636363",           # grey
    "forest": "#31a354",         # green
    "desert": "#ffcc00",         # yellow
    "mountain": "#d95f0e",       # brown
    "lake": "#1f78b4",           # blue
    "agriculture": "#e31a1c",    # red
}

# Marker shapes for each cluster
markers = {
    "city": "o",
    "forest": "s",
    "desert": "D",
    "mountain": "^",
    "lake": "P",
    "agriculture": "X",
}

fig, ax = plt.subplots(figsize=(8, 8))
for cluster, data in clusters.items():
    ax.scatter(data[:, 0], data[:, 1], c=colors[cluster], marker=markers[cluster], alpha=0.7, label=cluster)

# Draw annotation rectangle with dashed lines
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[1], annotaterecy[1]], color="black", lw=1, linestyle='--')
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[0], annotaterecy[0]], color="black", lw=1, linestyle='--')
ax.plot([annotaterecx[0], annotaterecx[0]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1, linestyle='--')
ax.plot([annotaterecx[1], annotaterecx[1]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1, linestyle='--')

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(insetaxes)  # Adjust the position to align with the right side of the main plot
for cluster, data in clusters.items():
    ax_inset.scatter(data[:, 0], data[:, 1], c=colors[cluster], marker=markers[cluster], alpha=0.7)
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")
ax_inset.set_title(inset_title)

ax.annotate("", xy=arrowstart, xytext=arrowend, textcoords="axes fraction", arrowprops=dict(facecolor="black", lw=0.1))

# Set labels and title
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)
ax.legend(ncol=3)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_45.pdf", bbox_inches="tight")
