
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generating new data for health domain (e.g., Age vs. Blood Pressure and Cholesterol Levels)
age_groups = {
    "20s": np.random.normal(loc=(120, 180), scale=10, size=(50, 2)),
    "30s": np.random.normal(loc=(125, 190), scale=10, size=(50, 2)),
    "40s": np.random.normal(loc=(130, 200), scale=10, size=(50, 2)),
    "50s": np.random.normal(loc=(135, 210), scale=10, size=(50, 2)),
    "60s": np.random.normal(loc=(140, 220), scale=10, size=(50, 2)),
    "70s": np.random.normal(loc=(145, 230), scale=10, size=(50, 2)),
}

# Text labels and other parameters
insetaxes = [0.65, 0.15, 0.3, 0.3]
insetxlim = [100, 150]
insetylim = [160, 250]
insetxticks = [110, 130, 150]
insetyticks = [170, 200, 230]
arrowstart = (130, 200)
arrowend = (0.6, 0.4)
annotaterecx = [120, 140]
annotaterecy = [180, 220]
xlabel = 'Blood Pressure (mm Hg)'
ylabel = 'Cholesterol Level (mg/dL)'
title = 'Health Data: Blood Pressure vs Cholesterol by Age Group'
legends = list(age_groups.keys())

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
# Colors for each age group (using a health-appropriate, professional palette)
colors = {
    "20s": "#1f77b4",  # muted blue
    "30s": "#ff7f0e",  # muted orange
    "40s": "#2ca02c",  # muted green
    "50s": "#d62728",  # muted red
    "60s": "#9467bd",  # muted purple
    "70s": "#8c564b",  # muted brown
}

fig, ax = plt.subplots(figsize=(8, 8))
for age_group, data in age_groups.items():
    ax.scatter(data[:, 0], data[:, 1], label=age_group, c=colors[age_group], alpha=0.6, edgecolors='w', s=100)

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)
ax.legend(title='Age Groups', loc='upper left')

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(insetaxes)  # Adjust the position to align with the right side of the main plot
for age_group, data in age_groups.items():
    ax_inset.scatter(data[:, 0], data[:, 1], c=colors[age_group], alpha=0.6, edgecolors='w', s=100)
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Drawing rectangle annotation
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Annotation with arrow
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.5),
)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_46.pdf", bbox_inches="tight")
