
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Updated gradient steps for simulated time (years)
years = np.linspace(0, 200, 50)

# Generate different economic trends for each line
gdp_growth = np.exp(0.01 * years)  # Exponential growth representing GDP
market_fluctuations = np.random.normal(loc=1.0, scale=0.3, size=len(years))  # Market noise
employment_rates = 0.01 * years + 50  # Linear growth representing employment rates
inflation_rates = np.log(years + 1) * 2  # Logarithmic trend representing inflation rates

# Simulate standard deviations for error bars
std_dev = 0.1
std_gdp = np.full_like(gdp_growth, std_dev)
std_market = np.full_like(market_fluctuations, std_dev)
std_employment = np.full_like(employment_rates, std_dev)
std_inflation = np.full_like(inflation_rates, std_dev)

# Labels and other text parameters
xlabel = "Years"
ylabel = "Value"
title = "Economic Trends Over Time"
legend_labels = ["Market Fluctuations", "GDP Growth"]
xticks = np.linspace(0, 200, 9)
yticks = np.arange(0, 550, 50)
axesinset = [0.16, 0.3, 0.3, 0.3]
insetxlim = [25, 75]
insetylim = [0.5, 2.5]
insetxticks = [25, 50, 75]
insetyticks = [0.5, 1.5, 2.5]
arrowstart = (50, 3.5)
arrowend = (0.274, 0.22)
annotaterecx = [25, 75]
annotaterecy = [1, 2]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(10, 6))  # Increased size for better visibility

# Plot the first trend (market fluctuations) on the main plot
ax.plot(years, market_fluctuations, 'o--', color='#FF7F0E', label=legend_labels[0])
ax.fill_between(years, market_fluctuations - std_market, market_fluctuations + std_market, color='#FF7F0E', alpha=0.2)

# Plot the second trend (GDP growth) on the main plot
ax.plot(years, gdp_growth, 's-', color='#1F77B4', label=legend_labels[1])
ax.fill_between(years, gdp_growth - std_gdp, gdp_growth + std_gdp, color='#1F77B4', alpha=0.2)

# Additional plots - employment rates and inflation rates 
ax.plot(years, employment_rates, 'v-.', color='#2CA02C', label="Employment Rates")
ax.fill_between(years, employment_rates - std_employment, employment_rates + std_employment, color='#2CA02C', alpha=0.2)

ax.plot(years, inflation_rates, 'd:', color='#9467BD', label="Inflation Rates")
ax.fill_between(years, inflation_rates - std_inflation, inflation_rates + std_inflation, color='#9467BD', alpha=0.2)

# Set labels, title, ticks, and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_title(title, fontsize=14)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.legend(loc="upper center",ncol=4,frameon=False, framealpha=0.9,bbox_to_anchor=(0.5, 1.2))
ax.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax.set_facecolor("#f9f9f9")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[1], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[0], annotaterecy[0]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[0]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[1], annotaterecx[1]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(axesinset)  # Adjust the position to align with the right side of the main plot

# Plot the market fluctuations on the inset
ax_inset.plot(years, market_fluctuations, 'o--', color='#FF7F0E', label=legend_labels[0])
ax_inset.fill_between(years, market_fluctuations - std_market, market_fluctuations + std_market, color='#FF7F0E', alpha=0.2)

# Plot the GDP growth on the inset
ax_inset.plot(years, gdp_growth, 's-', color='#1F77B4', label=legend_labels[1])
ax_inset.fill_between(years, gdp_growth - std_gdp, gdp_growth + std_gdp, color='#1F77B4', alpha=0.2)

# Set limits, ticks and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate("", xy=arrowstart, xytext=arrowend, textcoords="axes fraction", arrowprops=dict(facecolor="black", lw=0.1))

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_50.pdf", bbox_inches="tight")
