
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Updated days
days = np.linspace(0, 100, 50)

# Generate different trends for each health metric
daily_exercise_time = np.random.normal(loc=30, scale=5, size=len(days))  # Mean exercise time
caloric_intake = 2200 + 200 * np.sin(days * 0.2)  # Sinusoidal pattern for caloric intake
heart_rate = np.random.normal(loc=70, scale=5, size=len(days))  # Mean heart rate
steps_walked = np.linspace(5000, 10000, len(days)) + np.random.normal(loc=0, scale=300, size=len(days))  # Increasing trend for steps

# Simulate standard deviations for error
std_dev = 5
daily_exercise_time_std = np.full_like(daily_exercise_time, std_dev)
caloric_intake_std = np.full_like(caloric_intake, std_dev)
heart_rate_std = np.full_like(heart_rate, std_dev)
steps_walked_std = np.full_like(steps_walked, std_dev)

# Labels and configuration
xlabel = "Days"
ylabel = "Health Metrics"
line_labels = ["Daily Exercise Time", "Steps Walked"]
xticks = np.linspace(0, 100, 6)
yticks = np.arange(0, 12000, 2000)
axesinset = [0.3, 0.25, 0.3, 0.2]
insetxlim = [25, 75]
insetylim = [5000, 8000]
insetxticks = [25, 50, 75]
insetyticks = [6000, 7000, 8000]
arrowstart = (50, 6000)
arrowend = (0.274, 0.22)
annotaterecx = [25, 75]
annotaterecy = [5000, 8000]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(6, 6))

# Plot the first line on the main plot
ax.plot(days, daily_exercise_time, "o-", color="green", label=line_labels[0])
ax.fill_between(
    days,
    daily_exercise_time - daily_exercise_time_std,
    daily_exercise_time + daily_exercise_time_std,
    color="lightgreen",
    alpha=0.3,
)

# Plot the second line on the main plot
ax.plot(days, steps_walked, "s--", color="purple", label=line_labels[1])
ax.fill_between(
    days,
    steps_walked - steps_walked_std,
    steps_walked + steps_walked_std,
    color="lavender",
    alpha=0.3,
)

# Set labels, ticks, legend, and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax.set_facecolor("#f9f9f9")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(
    axesinset
)  # Adjust the position to align with the right side of the main plot

# Plot the first line on the inset
ax_inset.plot(days, daily_exercise_time, "o-", color="green", label=line_labels[0])
ax_inset.fill_between(
    days,
    daily_exercise_time - daily_exercise_time_std,
    daily_exercise_time + daily_exercise_time_std,
    color="lightgreen",
    alpha=0.3,
)

# Plot the second line on the inset
ax_inset.plot(days, steps_walked, "s--", color="purple", label=line_labels[1])
ax_inset.fill_between(
    days,
    steps_walked - steps_walked_std,
    steps_walked + steps_walked_std,
    color="lavender",
    alpha=0.3,
)

# Set limits, ticks, and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate(
    "",
    xy=arrowstart,  # Arrow start point (on the main plot)
    xytext=arrowend,  # Arrow end point (on the inset)
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_52.pdf", bbox_inches="tight")
