
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Updated years
years = np.linspace(1900, 2000, 50)

# Generate different trends for each line
annual_philosophical_works = np.random.normal(loc=50, scale=10, size=len(years))
popularity_philosophical_movements = (
    np.linspace(30, 80, len(years))
)

# Simulate standard deviations for error
std_dev = 5
annual_philosophical_works_std = np.full_like(annual_philosophical_works, std_dev)
popularity_philosophical_movements_std = np.full_like(popularity_philosophical_movements, std_dev)

# Labels and configuration
xlabel = "Years"
ylabel = "Philosophical Metrics"
line_labels = ["Annual Philosophical Works", "Popularity of Major Philosophical Movements"]
xticks = np.linspace(1900, 2000, 11)
yticks = np.arange(0, 100, 10)
axesinset = [0.6, 0.7, 0.2, 0.2]
insetxlim = [1950, 1980]
insetylim = [30, 60]
insetxticks = [1950, 1960, 1970, 1980]
insetyticks = [30, 40, 50, 60]
arrowstart = (1960, 60)
arrowend = (0.7, 0.7)
annotaterecx = [1950, 1980]
annotaterecy = [30, 60]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(8, 6))

# Plot the first line on the main plot
ax.plot(years, annual_philosophical_works, "-", color="#377eb8", label=line_labels[0], lw=2)

ax.fill_between(
    years,
    annual_philosophical_works - annual_philosophical_works_std,
    annual_philosophical_works + annual_philosophical_works_std,
    color="#a6cee3",
    alpha=0.5,
)

# Plot the second line on the main plot
ax.plot(years, popularity_philosophical_movements, "--", color="#4daf4a", label=line_labels[1], lw=2)

ax.fill_between(
    years,
    popularity_philosophical_movements - popularity_philosophical_movements_std,
    popularity_philosophical_movements + popularity_philosophical_movements_std,
    color="#b2df8a",
    alpha=0.5,
)

# Set labels, ticks, legend and grid for the main plot
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.set_ylim(0,120)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax.set_facecolor("#f0f0f0")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(
    axesinset
)

# Plot the first line on the inset
ax_inset.plot(years, annual_philosophical_works, "-", color="#377eb8", lw=1.5)

ax_inset.fill_between(
    years,
    annual_philosophical_works - annual_philosophical_works_std,
    annual_philosophical_works + annual_philosophical_works_std,
    color="#a6cee3",
    alpha=0.5,
)

# Plot the second line on the inset
ax_inset.plot(years, popularity_philosophical_movements, "--", color="#4daf4a", lw=1.5)

ax_inset.fill_between(
    years,
    popularity_philosophical_movements - popularity_philosophical_movements_std,
    popularity_philosophical_movements + popularity_philosophical_movements_std,
    color="#b2df8a",
    alpha=0.5,
)

# Set limits, ticks and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.5),
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_53.pdf", bbox_inches="tight")
