# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for sports teams
import numpy as np

np.random.seed(42)
sports_teams = {
    "TeamGryphons": np.random.normal(loc=(50, 80), scale=10, size=(50, 2)),
    "TeamFalcons": np.random.normal(loc=(60, 60), scale=10, size=(30, 2)),
    "TeamTigers": np.random.normal(loc=(70, 90), scale=10, size=(40, 2)),
    "TeamPanthers": np.random.normal(loc=(80, 50), scale=10, size=(60, 2)),
    "TeamWolves": np.random.normal(loc=(60, 40), scale=10, size=(70, 2)),
    "TeamLions": np.random.normal(loc=(40, 70), scale=10, size=(45, 2)),
}

title = "Sports Teams Performance Metrics"
xlabel = "Average Game Time (Minutes)"
ylabel = "Team Performance Score"
# Inset plot configuration
insetaxes = [0.2, 0.6, 0.3, 0.3]
insetxlim = [50, 70]  # Focus on TeamGryphons and TeamFalcons
insetylim = [60, 80]
insetxticks = [50.0, 60.0, 70.0]
insetyticks = [60.0, 70.0, 80.0]

# Arrow and annotation configuration
arrowstart = (50, 75)  # Arrow start for TeamGryphons
arrowend = (0.38, 0.7)  # Relative inset arrow end
annotaterecx = [50, 70]  # Annotation range for X
annotaterecy = [60, 80]  # Annotation range for Y

# Colors for sports teams
colors = {
    "TeamGryphons": "#1f77b4",
    "TeamFalcons": "#ff7f0e",
    "TeamTigers": "#2ca02c",
    "TeamPanthers": "#d62728",
    "TeamWolves": "#9467bd",
    "TeamLions": "#8c564b",
}

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 8))
for team, data in sports_teams.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[team],
        label=team,
        alpha=0.6,
        edgecolors="w",
        s=100,
    )

# Enclosing rectangle for annotation
rect = plt.Rectangle(
    (annotaterecx[0], annotaterecy[0]),
    annotaterecx[1] - annotaterecx[0],
    annotaterecy[1] - annotaterecy[0],
    linewidth=1,
    edgecolor="black",
    facecolor="none",
)
ax.add_patch(rect)

# Create the inset with zoomed-in view
ax_inset = fig.add_axes(insetaxes)
for team, data in sports_teams.items():
    ax_inset.scatter(
        data[:, 0], data[:, 1], c=colors[team], alpha=0.6, edgecolors="w", s=100
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=1),
)

# Adding titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_75.pdf", bbox_inches="tight")
