# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
tourist_attractions = {
    "landmarks": np.random.normal(loc=(600, 800), scale=100, size=(50, 2)),
    "beaches": np.random.normal(loc=(700, 700), scale=100, size=(30, 2)),
    "museums": np.random.normal(loc=(800, 900), scale=100, size=(40, 2)),
    "parks": np.random.normal(loc=(900, 600), scale=100, size=(60, 2)),
    "hotels": np.random.normal(loc=(1000, 500), scale=100, size=(70, 2)),
}

insetaxes = [0.15, 0.15, 0.2, 0.2]
insetxlim = [600, 900]
insetylim = [500, 1000]
insetxticks = [600, 750, 900]
insetyticks = [500, 750, 1000]
arrowstart = (620, 400)  # Focus area for annotation
arrowend = (0.35, 0.3)
annotaterecx = [600, 900]
annotaterecy = [500, 1000]
title = "Tourist Attraction Popularity & Visitors"
xlabel = "Popularity Score"
ylabel = "Visitor Frequency"
legendlabels = {
    "landmarks": "Landmarks",
    "beaches": "Beaches",
    "museums": "Museums",
    "parks": "Parks",
    "hotels": "Hotels",
}

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 10))

# Define colors and markers for each tourist attraction, with distinct and appealing colors
colors = {
    "landmarks": "#FF6347",  # Tomato
    "beaches": "#4682B4",  # SteelBlue
    "museums": "#32CD32",  # LimeGreen
    "parks": "#8A2BE2",  # BlueViolet
    "hotels": "#FFD700",  # Gold
}

markers = {
    "landmarks": "s",
    "beaches": "o",
    "museums": "^",
    "parks": "d",
    "hotels": "p",
}

for attraction, data in tourist_attractions.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[attraction],
        marker=markers[attraction],
        label=legendlabels[attraction],
        alpha=0.7,
    )

# Draw annotation rectangle around a specific region
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with a zoomed-in view of a specific region
ax_inset = fig.add_axes(insetaxes)
for attraction, data in tourist_attractions.items():
    ax_inset.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[attraction],
        marker=markers[attraction],
        alpha=0.7,
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add annotation to indicate the inset region
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# Add titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("PIP_80.pdf", bbox_inches="tight")
