
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Simulated data for crime rates over months
months = np.linspace(1, 12, 12)  # 12 months
crime_rates = np.array(
    [50, 55, 53, 52, 56, 60, 62, 63, 64, 65, 67, 70]
)
crime_rates_pred = crime_rates + np.random.normal(0, 3, len(crime_rates))

# Simulated data for Public Sentiment Index (PSI)
psi = np.linspace(70, 80, 12)
psi_pred = psi + np.random.uniform(-2, 2, len(psi))

# Extracted variables
crime_label = "Actual Crime Rate"
crime_pred_label = "Predicted Crime Rate"
psi_label = "Actual PSI"
psi_pred_label = "Predicted PSI"

crime_ylabel = "Crime Rates (per 1000 people)"
psi_xlabel = "Months"
psi_ylabel = "Public Sentiment Index (PSI)"
legend_labels = ["Actual", "Predicted"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
crime_ylim = [40, 80]
crime_xlim = [1, 12]
crime_yticks = [40, 50, 60, 70, 80]
crime_xticks = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]

psi_ylim = [65, 85]
psi_xlim = [1, 12]
psi_yticks = [65, 70, 75, 80, 85]
psi_xticks = crime_xticks

legend_loc = "lower center"
legend_bbox_to_anchor = (0.5, -0.2)
legend_ncol = 2
legend_frameon = False

# Create figure and axes
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(12, 12))

# Plot Crime Rates
ax1.plot(months, crime_rates, "o-", color="#1f77b4", label=crime_label, linewidth=2)
ax1.fill_between(months, crime_rates, color="#aec7e8", alpha=0.5)
ax1.plot(months, crime_rates_pred, "s-", color="#ff7f0e", label=crime_pred_label, linewidth=2)
ax1.fill_between(months, crime_rates_pred, color="#ffbb78", alpha=0.5)
ax1.set_ylim(crime_ylim)
ax1.set_xlim(crime_xlim)
ax1.set_yticks(crime_yticks)
ax1.set_xticks(crime_xticks)
ax1.set_ylabel(crime_ylabel)
ax1.tick_params(axis="both", which="both", length=0, labelsize=12)

# Plot PSI
ax2.plot(months, psi, "o-", color="#1f77b4", label=psi_label, linewidth=2)
ax2.fill_between(months, psi, color="#aec7e8", alpha=0.5)
ax2.plot(months, psi_pred, "s-", color="#ff7f0e", label=psi_pred_label, linewidth=2)
ax2.fill_between(months, psi_pred, color="#ffbb78", alpha=0.5)
ax2.set_ylim(psi_ylim)
ax2.set_xlim(psi_xlim)
ax2.set_yticks(psi_yticks)
ax2.set_xticks(psi_xticks)
ax2.set_xlabel(psi_xlabel, fontsize=14)
ax2.set_ylabel(psi_ylabel, fontsize=14)
ax2.tick_params(axis="both", which="both", length=0, labelsize=12)

# Create custom legend
blue_circle = mlines.Line2D([], [], color="#1f77b4", marker="o", markersize=6, label=legend_labels[0])
orange_square = mlines.Line2D([], [], color="#ff7f0e", marker="s", markersize=6, label=legend_labels[1])
plt.legend(handles=[blue_circle, orange_square], loc=legend_loc, bbox_to_anchor=legend_bbox_to_anchor, ncol=legend_ncol, frameon=legend_frameon)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("area_42.pdf", bbox_inches="tight")
