
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(42)

# Monthly labels
months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]

# Political approval and disapproval data (in percentages)
approval_ratings = np.random.normal(50, 10, size=12).clip(0, 100)
disapproval_ratings = 100 - approval_ratings

# Calculate cumulative values for the stacked area chart
cumulative_approval = approval_ratings
cumulative_disapproval = cumulative_approval + disapproval_ratings

# Positions for the bars on the x-axis
ind = np.arange(len(months))

# Variables for plot configuration
approval_label = "Approval"
disapproval_label = "Disapproval"
xlabel_text = "Month"
ylabel_text = "Percentage (%)"
title_text = "Monthly Political Approval and Disapproval Ratings"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xlim_values = (0, 11)
ylim_values = (0, 200)
yticks_values = list(range(0, 201, 20))
legend_location = "upper left"
legend_fontsize = 10
legend_frameon = True
legend_shadow = False
legend_facecolor = "#f9f9f9"
legend_ncol = 1
legend_bbox_to_anchor = (0.1, 1.0)

# Plot
fig, ax = plt.subplots(figsize=(12, 8))  # Adjusted for better aspect ratio
ax.fill_between(months, 0, cumulative_approval, label=approval_label, color="#1f77b4", alpha=0.7)
ax.fill_between(months, cumulative_approval, cumulative_disapproval, label=disapproval_label, color="#d62728", alpha=0.7)

# Enhancing the plot with additional visuals
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["left"].set_visible(True)
ax.spines["bottom"].set_visible(True)
ax.set_yticks(yticks_values)
# Setting the x-axis and y-axis limits dynamically
ax.set_ylim(*ylim_values)  # Ensure all data fits well
ax.set_xlim(*xlim_values)
# Labels, Title, and Grid
ax.set_xlabel(xlabel_text, fontsize=14)
ax.set_ylabel(ylabel_text, fontsize=14)
ax.set_title(title_text, fontsize=18, y=1.02)
ax.tick_params(axis="both", which="both", color="gray")
# Custom legend
ax.legend(
    loc=legend_location,
    fontsize=legend_fontsize,
    frameon=legend_frameon,
    shadow=legend_shadow,
    facecolor=legend_facecolor,
    ncol=legend_ncol,
    bbox_to_anchor=legend_bbox_to_anchor,
)

# Grid
ax.grid(True, linestyle="--", alpha=0.5, which="both")

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout to reduce white space
plt.tight_layout()
plt.savefig("area_65.pdf", bbox_inches="tight")
