
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Updated category names for the health domain
category_names = ["Very Fit", "Fit", "Moderately Fit", "Unfit", "Very Unfit"]
# Generate new data reflecting fitness levels in different cities
results = {
    "New York": [12, 18, 20, 28, 22],
    "Los Angeles": [22, 24, 30, 12, 12],
    "Chicago": [30, 34, 10, 4, 22],
    "Houston": [28, 12, 14, 18, 28],
    "Phoenix": [18, 32, 8, 8, 34],
    "Philadelphia": [10, 22, 10, 26, 32],
}
xlim = [-100, 100]
xticks = np.arange(-100, 101, 20)
xvline = 0
title = "Fitness Levels in Different Cities"
xlabel = "Percentage"
ylabel = "Cities"

def create_bar_chart(results, category_names):
    fig, ax = plt.subplots(figsize=(12, 8))

    labels = list(results.keys())
    data = np.array(list(results.values()))
    data_cum = data.cumsum(axis=1)
    middle_index = data.shape[1] // 2
    offsets = data[:, range(middle_index)].sum(axis=1) + data[:, middle_index] / 2

    # Optimized Color Mapping
    category_colors = plt.get_cmap("viridis")(np.linspace(0.1, 0.9, data.shape[1]))

    # Plot Bars
    for i, (colname, color) in enumerate(zip(category_names, category_colors)):
        widths = data[:, i]
        starts = data_cum[:, i] - widths - offsets
        rects = ax.barh(
            labels,
            widths,
            left=starts,
            height=0.5,
            label=colname,
            color=color,
            edgecolor="black",
        )

    # Add Zero Reference Line
    ax.axvline(xvline, linestyle="--", color="gray", alpha=0.7)

    # X Axis
    ax.set_xlim(xlim)
    ax.set_xticks(xticks)
    ax.xaxis.set_major_formatter(lambda x, pos: str(abs(int(x))))
    ax.set_xlabel(xlabel, fontsize=14)

    # Y Axis
    ax.set_ylabel(ylabel, fontsize=14)
    ax.invert_yaxis()

    # Remove spines
    ax.spines["right"].set_visible(False)
    ax.spines["top"].set_visible(False)
    ax.spines["left"].set_visible(False)

    # Legend
    ax.legend(ncol=len(category_names), bbox_to_anchor=(0.5, 1.1), loc="upper center", fontsize=12)

    # Set Background Color
    fig.set_facecolor("#F7F7F7")

    # Title
    ax.set_title(title, fontsize=16, weight='bold')

    return fig, ax

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = create_bar_chart(results, category_names)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("bar_252.pdf", bbox_inches="tight")
