
# ===================
# Part 1: Importing Libraries
# ===================
import colorsys
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulate population decline (%) in major cities during different periods
historical_population_decline = {
    f"Decline_Period_{period}": -np.abs(np.random.uniform(5, 15, size=4).round(2))
    for period in ["WW1", "WW2", "Great_Depression", "Cold_War"]
}

# Cities under consideration
cities = ["London", "Berlin", "New York", "Tokyo"]
ind = np.arange(4)
xlabel = "Population Decline Rate (%)"
title = "Historical Population Decline Rates by City and Period"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
def hsl_to_rgb(h, s, l):
    return colorsys.hls_to_rgb(h, l, s)


# Vintage color palette
colors = [
    hsl_to_rgb(hue, 0.7, 0.4) for hue in np.linspace(0.1, 0.9, 5)[:-1]
]

# Create the figure and axes objects
fig, ax = plt.subplots(figsize=(10, 6))

# The width of the bars
bar_width = 0.2

# Define some hatch patterns to use for bars
hatch_patterns = ["/", "\\", "x", "o"]

# Plotting data with randomization
for i, (label, decline_rates) in enumerate(historical_population_decline.items()):
    ax.barh(
        ind + bar_width * (i - 1.5),
        decline_rates,
        bar_width,
        label=label,
        color=colors[i],
        hatch=hatch_patterns[i % len(hatch_patterns)],
    )

# Adding labels, title, and custom y-axis tick labels, etc.
ax.set_xlabel(xlabel)
ax.set_title(title)
ax.set_yticks(ind)
ax.set_yticklabels(cities)

# Add random rotation to y-axis labels
for label in ax.get_yticklabels():
    label.set_rotation(np.random.randint(-30, 30))

# Adding legend
ax.legend(
    ncols=len(historical_population_decline.keys()), loc="upper center", bbox_to_anchor=(0.5, 1.15)
)

# Invert y-axis to have the first entry at the top
plt.gca().invert_yaxis()

# Randomly decide whether to show grid lines for x-axis and set their properties
if np.random.rand() > 0.5:
    ax.xaxis.grid(
        True,
        linestyle=np.random.choice(["--", ":", "-."]),
        color=np.random.rand(
            3,
        ),
        alpha=0.7,
    )

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("bar_277.pdf", bbox_inches="tight")
