
# ===================
# Part 1: Importing Libraries
# ===================
import colorsys
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Randomly generate annual healthcare expenditure growth rates
healthcare_expenditure_growth = {
    f"Expenditure_growth_{year}": np.random.randint(-10, -2, 4)
    for year in [2018, 2019, 2020, 2021]
}

countries = ["USA", "Canada", "UK", "Australia"]
ind = np.arange(4)
xlabel = "Healthcare Expenditure Growth Rate (%)"
title = "Annual Healthcare Expenditure Growth by Country and Year"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
def hsl_to_rgb(h, s, l):
    return colorsys.hls_to_rgb(h, l, s)

# Health-related color palette for each set of bars
colors = [
    hsl_to_rgb(hue, 0.5, 0.5) for hue in np.linspace(0.3, 0.7, 5)[:-1]
]  # Variations of green and blue

# Create the figure and axes objects
fig, ax = plt.subplots(figsize=(10, 6))

# The width of the bars
bar_width = 0.2

# Define some hatch patterns to use for bars
hatch_patterns = ["//", "xx", "oo", "++"]

# Plotting data with randomization
for i, (label, growth_rates) in enumerate(healthcare_expenditure_growth.items()):
    ax.barh(
        ind + bar_width * (i - 1.5),
        growth_rates,
        bar_width,
        label=label,
        color=colors[i],
        hatch=np.random.choice(hatch_patterns),
    )

# Adding labels, title, and custom y-axis tick labels, etc.
ax.set_xlabel(xlabel)
ax.set_title(title)
ax.set_yticks(ind)
ax.set_yticklabels(countries)

# Add random rotation to y-axis labels
for label in ax.get_yticklabels():
    label.set_rotation(np.random.randint(-30, 30))

# Adding legend
ax.legend(
    ncols=len(healthcare_expenditure_growth.keys()), loc="upper center", bbox_to_anchor=(0.5, 1.15)
)

# Invert y-axis to have the first entry at the top
plt.gca().invert_yaxis()

# Randomly decide whether to show grid lines for x-axis and set their properties
if np.random.rand() > 0.5:
    ax.xaxis.grid(
        True,
        linestyle=np.random.choice(["--", ":", "-."]),
        color=np.random.rand(
            3,
        ),
        alpha=0.7,
    )

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("bar_278.pdf", bbox_inches="tight")
