
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
import colorsys

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Redefining the data
countries = ["USA", "China", "Germany", "Japan", "UK", "France", "India"]
gdp_growth_rate = [2.3, 6.1, 0.5, 1.0, 1.5, 1.1, 7.0]

# Sorting the data in descending order while keeping track of the countries order
sorted_data = sorted(zip(gdp_growth_rate, countries), reverse=True)
sorted_gdp_growth_rate, sorted_countries = zip(*sorted_data)

title = "GDP Growth Rate by Country"
xlabel = "Country"
ylabel = "GDP Growth Rate (%)"
ylim = [0, np.max(sorted_gdp_growth_rate) + 2]

# Generate random colors with lower saturation
def hsl_to_rgb(h, s, l):
    return colorsys.hls_to_rgb(h, l, s)


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and bar chart with the sorted data
plt.figure(figsize=(12, 8))
# Randomly generate colors
colors = [hsl_to_rgb(hue, 0.5, 0.6) for hue in np.linspace(0, 1, len(countries) + 1)[:-1]]

bars = plt.bar(sorted_countries, sorted_gdp_growth_rate, color=colors)

# Add text labels on bars
for bar in bars:
    yval = bar.get_height()
    text_y = (
        yval - 0.5 if yval > 1 else yval + 0.2
    )  # Slight modification to avoid negative values
    plt.text(
        bar.get_x() + bar.get_width() / 2,
        text_y,
        f"{yval}%",
        ha="center",
        va="top" if text_y < yval else "bottom",
    )

# Set chart title and labels
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set y-axis range
plt.ylim(ylim)

# Add gridlines
plt.grid(axis="y", linestyle="--", alpha=0.7)

# Rotate x-axis labels
plt.xticks(rotation=45, fontsize=12)
plt.yticks(fontsize=12)

# Hide the top and right spines
plt.gca().spines["top"].set_visible(False)
plt.gca().spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Apply tight layout
plt.tight_layout()

plt.savefig("bar_280.pdf", bbox_inches="tight")
