
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
import colorsys

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Redefining the data
exhibits = ["Monet's Garden", "Van Gogh Alive", "Banksy Exhibit", "The Art of the Brick", "Yayoi Kusama", "Da Vinci Experience", "Picasso Blue"]
ratings = [82.0, 75.0, 65.0, 60.0, 90.0, 50.0, 78.0]

# Sorting the data in descending order while keeping track of the models order
sorted_data = sorted(zip(ratings, exhibits), reverse=True)
sorted_ratings, sorted_exhibits = zip(*sorted_data)

title = "Art Exhibit Ratings"
xlabel = "Exhibit"
ylabel = "Rating (%)"
ylim = [0, np.max(sorted_ratings) + 10]


# Generate colors inspired by art styles (e.g., pastel colors)
def hex_to_rgb(hex):
    hex = hex.lstrip('#')
    return tuple(int(hex[i:i+2], 16) / 255.0 for i in (0, 2, 4))



# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and bar chart with the sorted data
plt.figure(figsize=(12, 8))
color_palette = ["#FFB7C5", "#FFE5B4", "#D3FFCE", "#B0E0E6", "#C9C0BB", "#FFCCCB", "#F5FFFA"]
colors = [hex_to_rgb(color) for color in color_palette]

bars = plt.bar(sorted_exhibits, sorted_ratings, color=colors)

# Randomly decide where to put the text based on the value of the bar
for bar in bars:
    yval = bar.get_height()
    text_y = (
        yval - 5 if yval > 10 else yval + 1
    )  # Slight modification to avoid negative values
    plt.text(
        bar.get_x() + bar.get_width() / 2,
        text_y,
        f"{yval}%",
        ha="center",
        va="top" if text_y < yval else "bottom",
        fontsize=12, fontweight='bold', color='black'
    )

# Set chart title and labels
plt.title(title, fontsize=16, fontweight='bold')
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set y-axis range to a bit higher than the max value
plt.ylim(ylim)

# Set gridlines and ticks
plt.grid(axis="y", linestyle="--", alpha=0.7)

# Set tick rotation
plt.xticks(rotation=45, ha='right', fontsize=12)

# Hide the top and right spines
plt.gca().spines["top"].set_visible(False)
plt.gca().spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Apply tight layout
plt.tight_layout()

plt.savefig("bar_281.pdf", bbox_inches="tight")
