
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)

# Seed for reproducibility

# Categories in the domain of politics
categories = [
    "Voter Turnout",
    "Approval Ratings",
    "Campaign Spending",
    "Debate Performance",
    "Social Media Presence",
    "Media Coverage",
    "Policy Proposals",
]

# Increased number of layers with descriptive names
layer_data = {
    f"Region {chr(65+i)}": np.random.randint(10, 50, size=len(categories))
    for i in range(6)
}


# Hatch patterns to add texture to the bars
hatches = ["/", "\\", "|", "-", "+", "x"]

# Text labels and title
title = "Political Metrics Stacked Bar Chart"
xlabel = "Metrics"
ylabel = "Values"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 7))
# Color palette with political theme
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd", "#8c564b"]

bottoms = np.zeros(len(categories))

for i, (layer, values) in enumerate(layer_data.items()):
    bars = ax.bar(
        categories,
        values,
        bottom=bottoms,
        color=colors[i],
        label=layer,
        hatch=hatches[i]
    )

    # Add text annotations for readability
    for bar, bottom in zip(bars, bottoms):
        height = bar.get_height()
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            bottom + height / 2,
            f'{int(height)}',
            ha='center',
            va='center',
            fontsize=10,
            color="white",
            fontweight='bold'
        )

    # Update the bottoms for stacking
    bottoms += values

# Chart title and labels with enhanced styles
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=12)
plt.ylabel(ylabel, fontsize=12)

# Adjust legend to be at the top outside the plot area
plt.legend(
    title="Regions",
    title_fontsize="13",
    fontsize="11",
    loc="upper center",
    bbox_to_anchor=(0.5, 1.25),
    ncol=3
)

# Adding grid lines for better readability
ax.yaxis.grid(True, linestyle="--", which="major", color="grey", alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("bar_285.pdf", bbox_inches="tight")
