
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(1)  # Seed for reproducibility

# New categories representing famous attractions
categories = [
    "Beach Resort",
    "National Park",
    "Museum",
    "Historic Site",
    "Theme Park",
    "Zoo",
    "Aquarium",
]

# Visitors from different countries
layer_data = {
    f"Visitors from {country}": np.random.randint(1000, 5000, size=len(categories))
    for country in ["USA", "Canada", "UK", "Australia", "Germany", "France"]
}

title = "Visitor Statistics for Various Attractions"
xlabel = "Attractions"
ylabel = "Number of Visitors"

# Hatch patterns for visual differentiation
hatches = ["/", "\\", "|", "-", "+", "x"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Color palette for tourism-related visualization
colors = ["#4D8FAC", "#79B6C9", "#AACBEA", "#DEEBF7", "#BAE4B3", "#74C476"]

fig, ax = plt.subplots(figsize=(10, 7))

bottoms = np.array([0] * len(categories))

for i, (layer, values) in enumerate(layer_data.items()):
    bars = ax.bar(
        categories,
        values,
        bottom=bottoms,
        color=colors[i],
        label=layer,
        hatch=hatches[i],
    )

    # Adding annotations for better readability
    for bar, bottom in zip(bars, bottoms):
        height = bar.get_height()
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            bottom + height / 2,
            f"{height}",
            ha="center",
            va="center",
            color="black",
            fontsize=8,
            rotation=0,
        )

    bottoms += values

# Chart title and labels with enhanced styles
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=12)
plt.ylabel(ylabel, fontsize=12)

# Adjust legend to be at the top outside the plot area
plt.legend(
    title="Visitor Sources",
    title_fontsize="13",
    fontsize="11",
    loc="upper center",
    bbox_to_anchor=(0.5, 1.25),
    ncol=3,
)

# Adding grid lines for better readability
ax.yaxis.grid(True, linestyle="--", which="major", color="grey", alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("bar_286.pdf", bbox_inches="tight")
