
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# New sample data with different distributions
data1 = [
    np.random.normal(loc=1, scale=std, size=50)
    for std in np.linspace(1, 2, 7)
]
data2 = [
    np.random.normal(loc=1.5,scale=std, size=50)
    for std in np.linspace(1, 2.5, 7)
]
data3 = [
    np.random.normal(loc=2, scale=std, size=50)
    for std in np.linspace(1.5, 3, 7)
]

labels = [
    "Threshold",
    "Infeasible",
    "Method A",
    "Method B",
    "Method C",
]
xlabel = "Parameter Set"
ylabel = "Value Measurement"
ylim = [-10, 10]
axhlines = [-5, 5]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Boxplot
fig, ax = plt.subplots(figsize=(10, 7))

# Boxplot configurations
bp1 = ax.boxplot(
    data1,
    positions=np.array(range(len(data1))) * 2.0 - 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp2 = ax.boxplot(
    data2,
    positions=np.array(range(len(data2))) * 2.0,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp3 = ax.boxplot(
    data3,
    positions=np.array(range(len(data3))) * 2.0 + 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)

# Set properties for each boxplot
for bp, color in zip([bp1, bp2, bp3], ["#1f77b4", "#ff7f0e", "#2ca02c"]):
    for patch in bp["boxes"]:
        patch.set_facecolor(color)
    for whisker in bp["whiskers"]:
        whisker.set(color="black", linewidth=1)
    for cap in bp["caps"]:
        cap.set(color="black", linewidth=1)
    for median in bp["medians"]:
        median.set(color="black", linewidth=2)

# Max and min markers
box_data = [np.abs(box.get_path().vertices[1][1] - box.get_path().vertices[2][1]) for box in bp2["boxes"]]
max_pos = np.argmax(box_data)
min_pos = np.argmin(box_data)

ax.plot(
    max_pos * 2.0, bp2["medians"][max_pos].get_ydata()[0], 
    marker="o", color="red", markersize=10
)
ax.plot(
    min_pos * 2.0, bp2["medians"][min_pos].get_ydata()[0], 
    marker="o", color="blue", markersize=10
)

# Add dashed lines
for line in axhlines:
    ax.axhline(y=line, color="gray", linestyle="--", label="Threshold")

# Add legend
axhline_legend = mlines.Line2D([], [], color="gray", linestyle="--", label=labels[0])
marker_legend = mlines.Line2D(
    [], [], color="red", marker="o", linestyle="None", label=labels[1]
)
patch1 = mpatches.Patch(color="#1f77b4", label=labels[2])
patch2 = mpatches.Patch(color="#ff7f0e", label=labels[3])
patch3 = mpatches.Patch(color="#2ca02c", label=labels[4])

ax.legend(
    handles=[patch1, patch2, patch3, axhline_legend, marker_legend],
    loc="upper left",
    ncol=1,
    frameon=True,
    
)

# Set axis labels and limits
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title("Comparison of Methods")

# X-axis tick positions and labels
ax.set_xticks(range(0, len(data1) * 2, 2))
ax.set_xticklabels(["A", "B", "C", "D", "E", "F", "G"])

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("box_39.pdf", bbox_inches="tight")
