
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Patch

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Simulated data generation for soil moisture and air quality levels
temp = np.linspace(-10, 40, 400)
soil_moisture = np.linspace(0, 100, 400)
TEMP, SOIL_MOISTURE = np.meshgrid(temp, soil_moisture)
pos = np.dstack((TEMP, SOIL_MOISTURE))

# Simulate Gaussian distributions for soil moisture and air quality
def gaussian(x, y, mean, cov):
    return np.exp(
        -((x - mean[0]) ** 2 / (2 * cov[0][0]) + (y - mean[1]) ** 2 / (2 * cov[1][1]))
    )

Z_soil = gaussian(TEMP, SOIL_MOISTURE, [20, 50], [[50, 0], [0, 100]])
Z_air = gaussian(TEMP, SOIL_MOISTURE, [5, 75], [[40, 0], [0, 80]])

# Extracted variables
title = "Distribution of Soil Moisture and Air Quality Levels"
labels = ["Soil Moisture", "Air Quality"]
xlabel = "Temperature (°C)"
ylabel = "Soil Moisture (%)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
plt.figure(figsize=(10, 6))

# Contour plots for the distributions
contour_soil = plt.contourf(TEMP, SOIL_MOISTURE, Z_soil, cmap="Greens", alpha=0.6)
contour_air = plt.contourf(TEMP, SOIL_MOISTURE, Z_air, cmap="Purples", alpha=0.6)
plt.title(title)

# Create legend with color patches
legend_patches = [
    Patch(color="green", label=labels[0]),
    Patch(color="purple", label=labels[1]),
]
plt.legend(handles=legend_patches, loc='upper right')

# Axis labels
plt.xlabel(xlabel)
plt.ylabel(ylabel)

# Adjust plot to be visually appealing
plt.gca().set_aspect("equal", adjustable="box")

# ===================
# Part 4: Saving Output
# ===================
# Reduce whitespace around the plot
plt.tight_layout()
plt.savefig("contour_39.pdf", bbox_inches="tight")
