
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Patch

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulated data generation for traffic intensity and pollution levels
traffic = np.linspace(0, 100, 400)
pollution = np.linspace(0, 200, 400)
TRAFFIC, POLLUTION = np.meshgrid(traffic, pollution)
pos = np.dstack((TRAFFIC, POLLUTION))

# Simulate Gaussian distributions for traffic intensity and pollution
def gaussian(x, y, mean, cov):
    return np.exp(
        -((x - mean[0]) ** 2 / (2 * cov[0][0]) + (y - mean[1]) ** 2 / (2 * cov[1][1]))
    )

Z_traffic = gaussian(TRAFFIC, POLLUTION, [50, 100], [[200, 0], [0, 500]])
Z_pollution = gaussian(TRAFFIC, POLLUTION, [70, 150], [[150, 0], [0, 300]])

# Extracted variables
title = "Distribution of Traffic Intensity and Pollution Levels"
labels = ["Traffic Intensity", "Pollution Levels"]
xlabel = "Traffic Intensity (vehicles/hour)"
ylabel = "Pollution Levels (ppm)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
plt.figure(figsize=(10, 6))

# Contour plots for the distributions
contour_traffic = plt.contourf(TRAFFIC, POLLUTION, Z_traffic, cmap="Reds", alpha=0.6)
contour_pollution = plt.contourf(TRAFFIC, POLLUTION, Z_pollution, cmap="Greens", alpha=0.6)
plt.title(title)

# Create legend with color patches
legend_patches = [
    Patch(color="red", label=labels[0]),
    Patch(color="green", label=labels[1]),
]
plt.legend(handles=legend_patches)

# Axis labels
plt.xlabel(xlabel)
plt.ylabel(ylabel)

# Adjust plot to be visually appealing
plt.gca().set_aspect("equal", adjustable="box")

# ===================
# Part 4: Saving Output
# ===================
# Reduce whitespace around the plot
plt.tight_layout()
plt.savefig("contour_40.pdf", bbox_inches="tight")
