
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Create a grid of x (investment amount) and z (return rate) values
np.random.seed(2)  # Different seed for different data
x = np.linspace(0, 10, 150)
z = np.linspace(0, 10, 150)
X, Z = np.meshgrid(x, z)

# Define the landscape function with different peaks for economic data
def landscape_function(X, Z):
    return (
        np.exp(-((X - 2) ** 2 + (Z - 3) ** 2) / 1.5)
        + np.exp(-((X - 6) ** 2 + (Z - 7) ** 2) / 2)
        + np.exp(-((X - 8) ** 2 + (Z - 2) ** 2) / 0.5)
    )

# Calculate the function values on the grid
Z_values = landscape_function(X, Z)
xlabel = "Investment Amount (in thousands)"
ylabel = "Return Rate (percentage)"
title = "Economic Performance Landscape"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the contour plot
plt.figure(figsize=(12, 10))

# Using a colormap suitable for economic data
n_colors = 15  # Number of discrete colors in the colormap
discrete_cmap = plt.cm.get_cmap("viridis", n_colors)

# Create contour plot
contour = plt.contourf(X, Z, Z_values, levels=n_colors, cmap=discrete_cmap)

# Add a color bar
cbar = plt.colorbar(
    contour, ticks=np.linspace(Z_values.min(), Z_values.max(), n_colors)
)

# Label the axes
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)

# Add grid lines for better readability
plt.grid(True, linestyle='--', alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()

# Show the plot
plt.savefig("contour_53.pdf", bbox_inches="tight")
