
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data configuration
np.random.seed(42)

# Grid
x = np.linspace(-15, 15, 300)
y = np.linspace(-15, 15, 300)
X, Y = np.meshgrid(x, y)

# Irrigation point locations
irrigation_points = [(5, -5), (-5, 5), (-10, -10), (10, 10), (0, -15)]

# Influence strength function for soil moisture
def moisture_influence(x, y, points):
    Z = np.zeros_like(x)
    for px, py in points:
        Z += np.exp(-((x - px) ** 2 + (y - py) ** 2) / 20)
    return Z

# Calculate Z as influence strength
Z = moisture_influence(X, Y, irrigation_points)

# Text variables
title_text = "Soil Moisture Distribution"
xlabel_text = "X Coordinate (meters)"
ylabel_text = "Y Coordinate (meters)"
colorbar_label_text = "Moisture Level"
irrigation_label_format = "Irrigation Point ({},{})"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
legend_location = "upper right"
contour_label_format = "%.1f"
fig, ax = plt.subplots(figsize=(10, 10))

# Contour
cnt = ax.contour(X, Y, Z, cmap="viridis", linewidths=1)
contour_filled = ax.contourf(X, Y, Z, cmap="viridis", alpha=0.7)
ax.clabel(cnt, cnt.levels, inline=True, fontsize=10, fmt=contour_label_format)

# Irrigation point locations
for px, py in irrigation_points:
    ax.plot(px, py, "g^", markersize=12, label=irrigation_label_format.format(px, py))

# Adding color bar
cbar = fig.colorbar(contour_filled, ax=ax)
cbar.set_label(colorbar_label_text)

# Titles and labels
ax.set_title(title_text)
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.legend(loc=legend_location)

# Grid for better readability
ax.grid(True, which='both', linestyle='--', linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("contour_70.pdf", bbox_inches="tight")
